/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.util;

/**
 * 施設園芸制御に用いられる評価値計算機ユーティリティです。
 * @author WaBit
 */
public class HortCalculator {

    // 摂氏0度の絶対温度
    private static final double ZERO_C = 273.15;

    private HortCalculator(){

    }

    /**
     * 飽和水蒸気圧を計算します
     * @param temp セ氏温度
     * @return 飽和水蒸気圧(Pa)
     */
    public static double getSWVP(double temp) {
        double atemp = temp + ZERO_C;
        return Math.pow(2.71828182845904,
            (-6096.9385 * Math.pow(atemp, -1)
            + 21.2409642
            - 2.711193 * Math.pow(10,-2) * atemp
            + 1.673952 * Math.pow(10,-5) * Math.pow(atemp,2)
            + 2.433502 * Math.log(atemp)));
    }

    /**
     * 水蒸気圧を計算します。
     * @param temp セ氏温度
     * @param humid 相対湿度
     * @return 水蒸気圧(Pa)
     */
    public static double getWVP(double temp, double humid) {
        return getSWVP(temp) * humid / 100.0;
    }

    /**
     * 飽和水蒸気量(容積絶対湿度)を計算します。
     * @param temp セ氏温度
     * @param humid 相対湿度
     * @return 絶対湿度 (g/m^3)
     */
    public static double getAbsoluteHumidity(double temp, double humid) {
        // 飽和水蒸気量計算
        return 2.17 * getWVP(temp, humid) / (temp + ZERO_C);
    }

    /**
     * 飽差を計算します。
     * ここでいう飽差とは、空気中1立法メートル当たりあとどれぐらいの水分(g)を含むことができるかを指します。<br />
     * VPD (Vapour Pressure Dificit) [単位hPa] とは異なるので注意。
     * @param temp セ氏温度
     * @param humid 相対湿度
     * @return 飽差 (g/m^3)
     */
    public static double getHumidityDeficit(double temp, double humid) {
        return 2.17 * ((100 - humid)/ 100 * getSWVP(temp) / (temp + ZERO_C));
    }

    /**
     * 露点を計算します。
     * @param temp セ氏温度
     * @param humid 相対湿度
     * @return 露点(℃)
     */
    public static double getDewPoint(double temp, double humid) {
        double wvp = getWVP(temp, humid);
        return 13.715 * Math.log(wvp / 611.213)
                + 8.4262 / 10 * Math.pow(Math.log(wvp/611.213), 2)
                + 1.9048 / 100 * Math.pow(Math.log(wvp/611.213), 3)
                + 7.8158 / 1000 * Math.pow(Math.log(wvp/611.213), 4);
    }


}
