/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.protocol;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.SortedMap;
import java.util.TreeMap;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.Attributes;

import com.wabit.uecs.Ccm;
import com.wabit.uecs.CcmID;
import com.wabit.uecs.CcmLevel;
import com.wabit.uecs.UecsConstants;

/**
 * 通信子スキャン応答CCMクラスです。
 * @author WaBit
 *
 */
public class CcmScanReplyCcm extends XmlCcm {

    /** MACタグ名 */
    public static final String TAG_CCMNUM = "CCMNUM";

    /** MACタグ名 */
    public static final String TAG_CCM = "CCM";

    /** page属性名 */
    public static final String ATTR_PAGE = "page";

    /** total属性名 */
    public static final String ATTR_TOTAL = "total";

    /** No属性名 */
    public static final String ATTR_NO = "No";

    /** cast属性名 */
    public static final String ATTR_CAST = "cast";

    /** unit属性名 */
    public static final String ATTR_UNIT = "unit";

    /** SR属性名 */
    public static final String ATTR_SR = "SR";

    /** LV属性名 */
    public static final String ATTR_LV = "LV";

    private SortedMap<Integer, Ccm> ccmMap =
            Collections.synchronizedSortedMap(new TreeMap<Integer, Ccm>());
    private int page = 1;
    private int total = 1;
    private Ccm lastCcm;

    /**
     * リストのページ番号を取得します。
     * @return ページ番号
     */
    public int getPage() {
        return page;
    }

    /**
     * ページの総数を取得します。
     * @return 総ページ数
     */
    public int getTotal() {
        return total;
    }

    /**
     * リストのページ番号を設定します。
     * @param page ページ番号
     */
    public void setPage(int page) {
        this.page = page;
    }

    /**
     * ページの総数を設定します。
     * @param total 総ページ数
     */
    public void setTotal(int total) {
        this.total = total;
    }

    /**
     * CCM要素をリストに追加します。
     * 既登録であれば入れ替わります。
     * @param no CCM通し番号
     * @param ccm CCMオブジェクト。
     */
    public void putCcm(int no, Ccm ccm) {
        ccmMap.put(no, ccm);
    }

    /**
     * CCMをリストから除外します。
     * @param no 通し番号
     * @return 削除されたCCM
     */
    public Ccm removeCcm(int no) {
        return ccmMap.remove(no);
    }

    /**
     * CCM定義を取得します。
     * @param no CCM通し番号
     * @return CCM
     */
    public Ccm getCcm(int no) {
        return ccmMap.get(new Integer(no));
    }

    /**
     * CCM番号一覧を取得します。
     * @return CCM番号リスト
     */
    public List<Integer> listCcmNo() {
        return new ArrayList<Integer>(ccmMap.keySet());
    }

    @Override
    void appendXml(StringBuilder sb) {
        sb.append("<").append(TAG_CCMNUM);
        sb.append(" page=\"").append(page);
        sb.append("\" total=\"").append(total).append("\">");
        sb.append(ccmMap.size());
        sb.append("</").append(TAG_CCMNUM).append(">");
        Iterator<Integer> it = ccmMap.keySet().iterator();
        while (it.hasNext()) {
            Integer no = it.next();
            Ccm data = ccmMap.get(no);
            sb.append("<").append(TAG_CCM);
            sb.append(" No=\"").append(no).append("\"");
            sb.append(" room=\"").append(data.getRoom()).append("\"");
            sb.append(" region=\"").append(data.getRegion()).append("\"");
            sb.append(" order=\"").append(data.getOrder()).append("\"");
            sb.append(" priority=\"").append(data.getPriority()).append("\"");
            sb.append(" cast=\"").append(data.getCast()).append("\"");
            sb.append(" unit=\"").append(escape(data.getUnit())).append("\"");
            sb.append(" SR=\"").append(escape(data.getSide())).append("\"");
            sb.append(" LV=\"").append(escape(data.getLevel())).append("\"");
            sb.append(">");
            sb.append(escape(data.getType()));
            sb.append("</").append(TAG_CCM).append(">");
        }
    }

    @Override
    void importXml(Document doc) throws Exception {
        Element elm = (Element) doc.getElementsByTagName(TAG_CCMNUM).item(0);
        // UECS規約では、すべて省略不可のため、nullや空チェックはしない
        String val = elm.getAttribute(ATTR_PAGE);
        setPage(Integer.parseInt(val));
        val = elm.getAttribute(ATTR_TOTAL);
        setTotal(Integer.parseInt(val));

        NodeList list = doc.getElementsByTagName(TAG_CCM);
        for (int i = 0; i < list.getLength(); i++) {
            elm = (Element) list.item(i);
            int no = Integer.parseInt(elm.getAttribute(ATTR_NO));

            Ccm ccmData = new Ccm();
            ccmData.setCast(Integer.parseInt(elm.getAttribute(ATTR_CAST)));
            ccmData.setUnit(elm.getAttribute(ATTR_UNIT));

            val = elm.getAttribute(ATTR_ROOM);
            ccmData.setRoom(Integer.parseInt(val));
            val = elm.getAttribute(ATTR_REGION);
            ccmData.setRegion(Integer.parseInt(val));
            val = elm.getAttribute(ATTR_ORDER);
            ccmData.setOrder(Integer.parseInt(val));
            val = elm.getAttribute(ATTR_PRIORITY);
            ccmData.setPriority(Integer.parseInt(val));
            val = elm.getAttribute(ATTR_SR);
            ccmData.setSide(val.charAt(0));
            val = elm.getAttribute(ATTR_LV);
            ccmData.setLevel(new CcmLevel(val));
            val = elm.getFirstChild().getNodeValue();
            ccmData.setType(new CcmID(val));

            this.putCcm(no, ccmData);
        }

    }

    @Override
    public int getPort() {
        return UecsConstants.SCAN_PORT;
    }

    @Override
    void setXmlAttributes(String tagName, Attributes attributes) {
        if (TAG_CCMNUM.equals(tagName)) {
            // UECS規約では、すべて省略不可のため、nullや空チェックはしない
            String val = attributes.getValue(ATTR_PAGE);
            setPage(Integer.parseInt(val));
            val = attributes.getValue(ATTR_TOTAL);
            setTotal(Integer.parseInt(val));
        } else if (TAG_CCM.equals(tagName)) {
            int no = Integer.parseInt(attributes.getValue(ATTR_NO));
            Ccm ccmData = new Ccm();
            ccmData.setCast(Integer.parseInt(attributes.getValue(ATTR_CAST)));
            ccmData.setUnit(attributes.getValue(ATTR_UNIT));
            String val = attributes.getValue(ATTR_ROOM);
            ccmData.setRoom(Integer.parseInt(val));
            val = attributes.getValue(ATTR_REGION);
            ccmData.setRegion(Integer.parseInt(val));
            val = attributes.getValue(ATTR_ORDER);
            ccmData.setOrder(Integer.parseInt(val));
            val = attributes.getValue(ATTR_PRIORITY);
            ccmData.setPriority(Integer.parseInt(val));
            val = attributes.getValue(ATTR_SR);
            ccmData.setSide(val.charAt(0));
            val = attributes.getValue(ATTR_LV);
            ccmData.setLevel(new CcmLevel(val));
            lastCcm = ccmData;
            this.putCcm(no, ccmData);
        }

    }

    @Override
    void setXmlValue(String tagName, String value) {
        if (TAG_CCM.equals(tagName)) {
            lastCcm.setType(new CcmID(value));
        }
    }

}
