/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.device.actuator;

import java.util.Calendar;

/**
 * 定周期スイッチ(ON/OFF)実行アクションです。
 *
 * @author WaBit
 *
 */
public class PeriodicSwitchAction extends AbstractCyclicAction {

    private long periodicInterval;
    private long onInterval;
    private long lastOnTime;
    // 無効時間帯
    private int inactiveStart = -1;
    private int inactiveEnd = -1;

    /**
     * デフォルトインスタンスを取得します
     */
    public PeriodicSwitchAction() {
        // 1秒間隔で動作チェックする
        super(1000L);
    }

    /**
     * 動作周期時間を取得します。
     * @return 周期間隔(msec)
     */
    public long getPeriodicInterval() {
        return periodicInterval;
    }

    /**
     * 動作周期時間を設定します。
     * @param periodicInterval 周期時間
     */
    public void setPeriodicInterval(long periodicInterval) {
        this.periodicInterval = periodicInterval;
    }

    /**
     * ON状態を維持する時間を取得します。
     * @return 維持間隔(msec)
     */
    public long getOnInterval() {
        return onInterval;
    }

    /**
     * ON状態を維持する時間を設定します。
     * @param interval 維持間隔(msec)
     */
    public void setOnInterval(long interval) {
        this.onInterval = interval;
    }

    /**
     * 強制動作停止時間帯（開始）を設定します。
     * @param hour 時
     * @param min 分
     */
    public void setInactiveStartTime(int hour, int min) {
        this.inactiveStart = hour * 10000 + min * 100;
    }

    /**
     * 強制動作停止時間帯（終了）を設定します。
     * @param hour 時
     * @param min 分
     */
    public void setInactiveEndTime(int hour, int min) {
        this.inactiveEnd = hour * 10000 + min * 100;
    }

    /**
     * 設定された周期時間に従って、ON/OFF切り替え動作します。
     */
    @Override
    public void execute(ActuatorComponent<?> component) throws Exception {
        // 無効時間帯判定
        if (inactiveStart >= 0 && inactiveEnd >= 0) {
            Calendar cal = Calendar.getInstance();
            int time = cal.get(Calendar.HOUR_OF_DAY) * 10000
                        + cal.get(Calendar.MINUTE) * 100
                        + cal.get(Calendar.SECOND);

            // 無効時間内は以降の処理を行わない
            if (inactiveStart < inactiveEnd) {
                if (time >= inactiveStart && time <= inactiveEnd) {
                    return;
                }
            } else if (inactiveStart > inactiveEnd) {
                // 日またぎ時間帯
                if ((time >= inactiveStart && time <= 235959)
                    || (time >= 000000 && time <= inactiveEnd)){
                    return;
                }
            }
        }

        long current = System.currentTimeMillis();
        long delay = current - lastOnTime;
        if (delay >= periodicInterval) {
            if (lastOnTime == 0 || component.getValue() == null || component.getValue().intValue() != 1) {
                component.setValue(1);
            }
            lastOnTime = current;
        } else if (delay > onInterval) {
            if (component.getValue() == null || component.getValue().intValue() != 0) {
                component.setValue(0);
            }
        }
    }


}
