/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.device;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * 機器コンポーネント基底実装です。
 * CCMとコンポーネントを対応付けするための共通実装が含まれます。
 *
 * @author WaBit
 */
public abstract class AbstractComponent<T extends ComponentConfig> implements IComponent<T> {

    private Set<IComponentListener> listeners = Collections.synchronizedSet(new HashSet<IComponentListener>());

    private String id;

    private Number value;

    private IDevice<?> device;

    private T config;

    /**
     * コンストラクタ.
     * @param id コンポーネントID
     * @param config 設定値オブジェクト
     */
    public AbstractComponent(String id, T config) {
        this.id = id;
        this.config = config;
    }

    /*
     * (non-Javadoc)
     * @see com.wabit.uecs.device.IComponent#getConfig()
     */
    @Override
    public T getConfig() {
        return config;
    }

    /*
     * (non-Javadoc)
     * @see com.wabit.uecs.device.IComponent#getName()
     */
    @Override
    public String getId() {
        return id;
    }

    /*
     * (non-Javadoc)
     * @see com.wabit.uecs.device.IComponent#getValue()
     */
    @Override
    public Number getValue() {
        return value;
    }

    /*
     * (non-Javadoc)
     * @see com.wabit.uecs.device.IComponent#setValue(java.lang.Number)
     */
    @Override
    public void setValue(Number value) {
        this.value = value;
        notifyUpdated();
    }

    /*
     * (non-Javadoc)
     * @see com.wabit.uecs.device.IComponent#getDevice()
     */
    @Override
    public IDevice<?> getDevice() {
        return this.device;
    }

    /**
     * 所属するデバイスを登録します。
     * @param device デバイスインスタンス
     */
    protected void setDevice(IDevice<?> device) {
        this.device = device;
    }

    @Override
    public void addListener(IComponentListener listener) {
        listeners.add(listener);
    }

    @Override
    public List<IComponentListener> getListeners() {
        return new ArrayList<IComponentListener>(listeners);
    }

    @Override
    public boolean removeListener(IComponentListener listener) {
        return listeners.remove(listener);
    }

    /*
     * (non-Javadoc)
     * @see com.wabit.uecs.device.IComponent#init(com.wabit.uecs.device.IDevice)
     */
    @Override
    public void init(IDevice<?> device) throws Exception {
        try {
            setDevice(device);
            onInit();
        } catch (Exception e) {
            notifyException(e);
            throw e;
        }
    }

    /*
     * (non-Javadoc)
     * @see com.wabit.uecs.device.IComponent#start()
     */
    @Override
    public void start() throws Exception {
        try {
            onStart();
            notifyStarted();
        } catch (Exception e) {
            notifyException(e);
            throw e;
        }
    }

    /*
     * (non-Javadoc)
     * @see com.wabit.uecs.device.IComponent#stop()
     */
    @Override
    public void stop() throws Exception {
        try {
            onStop();
            notifyStopped();
        } catch (Exception e) {
            notifyException(e);
            throw e;
        }
    }

    /**
     * リスナーに開始イベントを伝達します。
     */
    protected void notifyStarted() {
        for (IComponentListener listener : listeners) {
            listener.componentStarted(this);
        }
    }

    /**
     * リスナーに停止イベントを伝達します。
     */
    protected void notifyStopped() {
        for (IComponentListener listener : listeners) {
            listener.componentStopped(this);
        }
    }

    /**
     * リスナーに更新イベントを伝達します。
     */
    protected void notifyUpdated() {
        for (IComponentListener listener : listeners) {
            listener.componentValueUpdated(this);
        }
    }

    /**
     * リスナーに例外発生を通知します。
     * @param e
     */
    protected void notifyException(Exception e) {
        for (IComponentListener listener : listeners) {
            listener.handleComponentException(this, e);
        }
    }

    /**
     * 初期化(init)実行時に呼び出されます。
     * 主に内部コンポーネントの作成と登録処理を実装します。
     *
     * @throws Exception 処理に失敗するとスローされます。
     */
    abstract protected void onInit() throws Exception;

    /**
     * 起動時の追加処理を実装するメソッドです。
     *
     * @throws Exception 処理に失敗するとスローされます。
     */
    abstract protected void onStart() throws Exception;

    /**
     * 停止時の追加処理を実装するメソッドです。
     *
     * @throws Exception 処理に失敗するとスローされます。
     */
    abstract protected void onStop() throws Exception;
}
