/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.Inet4Address;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import com.wabit.uecs.ccm.CndCcm;
import com.wabit.uecs.device.IDevice;
import com.wabit.uecs.protocol.DataCcm;

/**
 * UECS準拠ノードの基本動作を実装した抽象クラスです。
 * 一般的な具象UECSノードは本クラスを継承して作成することができます。
 *
 * @author WaBit
 *
 */
public abstract class AbstractUecsNode<T extends NodeConfig> implements IUecsNode<T> {

    // 受信タスク
    private class ReceiveTask implements Runnable {
        private IUecsProtocolHandler handler;

        private UecsRequest req;

        private ReceiveTask(IUecsProtocolHandler handler, byte[] data, InetAddress from) {
            this.handler = handler;
            this.req = new UecsRequest(from, handler.getPort(), data, System.currentTimeMillis());
        }

        @Override
        public void run() {
            UecsResponse res = new UecsResponse();
            try {
                handler.handle(req, res);
                if (res.isValid()) {
                    sendPacket(res.getAddress(), res.getPort(), res.getData());
                }
            } catch (Exception e) {
                // CCM受信エラー
                onStatus(UecsConstants.STATUS_ATTENTION_CCM_RECV_ERROR);
                handler.handleException(req, res, e);
            } finally {
            }
        }

    }

    // 周期動作タスク
    private class CycleTask implements Runnable {
        private CcmService sv;

        private CycleTask(CcmService def) {
            this.sv = def;
        }

        /**
         * CCMの定周期送信で実行されます。
         */
        @Override
        public void run() {
            // CCMが無効の場合は動作しない
            if (sv.isEnable()) {
                Ccm ccm = sv.getCcm();
                if (UecsConstants.SENDER == ccm.getSide()) {
                    // 周期送信
                    if (ccm.getNumberValue() != null) {
                        try {
                            DataCcm dccm = sv.createDataCcm(ccm);
                            sendPacket(getBroadcastAddress(), dccm.getPort(), dccm.toXmlBytes());
                            // CCM送信イベント通知
                            for (ICcmServiceListener lsn : sv.getListeners()) {
                                lsn.ccmSent(sv, dccm);
                            }
                            offStatus(UecsConstants.STATUS_ALART_COMMUNICATION_ERROR);
                        } catch (Exception e) {
                            onStatus(UecsConstants.STATUS_ALART_COMMUNICATION_ERROR);
                            notifyException(e);
                        }
                    }
                } else if (UecsConstants.RECEIVER == ccm.getSide()) {
                    // 受信CCM有効期限をチェック
                    sv.checkExpiration();
                }
            }
        }

    }

    /*
     * UDP実装 内部サーバクラス。
     */
    private class UdpServer implements Runnable {

        private DatagramSocket receiveSocket;

        private Thread thread;

        private IUecsProtocolHandler handler;

        /**
         * コンストラクタ
         *
         * @param port
         */
        public UdpServer(IUecsProtocolHandler handler) {
            this.handler = handler;
        }

        /*
         * (non-Javadoc)
         *
         * @see java.lang.Runnable#run()
         */
        @Override
        public void run() {
            byte[] buffer = new byte[UecsConstants.UDP_PACKET_SIZE];
            DatagramPacket packet = new DatagramPacket(buffer, UecsConstants.UDP_PACKET_SIZE);
            while (true) {
                try {
                    receiveSocket.receive(packet);

                    //TODO 暫定高負荷対策
                    //if (taskCount >= 100) {
                    //    continue;
                    //}

                    int start = packet.getOffset();
                    int len = packet.getLength();

                    // 一部他メーカのデータは末端が'\0'で埋められている
                    // のでその対策。本来はおかしい
                    for (int i = len - 1; i >= 0; i--) {
                        if (buffer[i] == '\0') {
                            len--;
                        } else {
                            break;
                        }
                    }

                    byte[] data = new byte[len];
                    System.arraycopy(buffer, start, data, 0, len);

                    InetAddress address = packet.getAddress();

                    // ハンドラを起動する
                    eventExecutor.execute(new ReceiveTask(handler, data, address));
                    offStatus(UecsConstants.STATUS_ATTENTION_CCM_RECV_ERROR);

                } catch (SocketException e) {
                    // クローズが呼ばれた
                    break;
                } catch (Exception e) {
                    // CCM受信エラー
                    onStatus(UecsConstants.STATUS_ATTENTION_CCM_RECV_ERROR);
                    notifyException(e);
                }
            }
        }

        /**
         * サービスを開始します。
         */
        public synchronized void start() {
            try {
                // 255.255.255.255のブロードキャスト受信のためにワイルドカードアドレスにバインドする
                // NIC二枚以上あると送信元IPが別サブネットになることがある.
                // 現時点で対策なし
                receiveSocket = new DatagramSocket(handler.getPort());
                receiveSocket.setReuseAddress(true);

                thread = new Thread(this);
                thread.start();
            } catch (SocketException e) {
                onStatus(UecsConstants.STATUS_ALART_INIT_FAILED);
                notifyException(e);
                throw new NodeRuntimeException(e);
            }

        }

        /**
         * サービスを停止します。
         */
        public synchronized void stop() {
            if (receiveSocket != null && !receiveSocket.isClosed()) {
                receiveSocket.close();
            }
            receiveSocket = null;
            thread = null;
        }

    }

    // 設定項目キー : 送信時刻を平準化するための遅延時間
    private static final String DELAY_TIME = "delay_time";

    // デフォルトのCCM送信ずらし時間
    private static final long DEFAULT_DELAY = 50L; // 50msec;

    private String uecsVersion;

    private InetAddress broadcastAddress;

    private InetAddress ipAddress;

    private byte[] macAddress;

    private String vender;

    private String uecsID;

    private DatagramSocket sendSocket = null;

    private T config;

    private String name;

    private boolean isActive = false;

    private int status = UecsConstants.STATUS_NORMAL;

    // ノード状態CCM
    private CcmService cndCcmService;

    // 定期CCMチェックスレッド
    private ScheduledExecutorService cycleExecutor;

    // イベント処理スレッド
    private ExecutorService eventExecutor;

    // 登録CCMリスト
    private List<CcmService> ccmList = Collections.synchronizedList(new ArrayList<CcmService>());

    // CCM受信サーバリスト
    private List<UdpServer> servers = Collections.synchronizedList(new ArrayList<UdpServer>());

    // プロトコルハンドラリスト
    private List<IUecsProtocolHandler> protocols = Collections.synchronizedList(new ArrayList<IUecsProtocolHandler>());

    // デバイスリスト
    private Map<String, IDevice<?>> deviceMap = Collections.synchronizedMap(new LinkedHashMap<String, IDevice<?>>());

    // リスナー
    private Set<IUecsNodeListener> listeners = Collections.synchronizedSet(new HashSet<IUecsNodeListener>());

    /**
     * コンストラクタ.
     * @param conf 設定情報オブジェクト
     */
    public AbstractUecsNode(T conf) {
        this.config = conf;
    }

    @Override
    public void addListener(IUecsNodeListener listener) {
        listeners.add(listener);
    }

    @Override
    public List<IUecsNodeListener> getListeners() {
        return new ArrayList<IUecsNodeListener>(listeners);
    }

    @Override
    public boolean removeListener(IUecsNodeListener listener) {
        return listeners.remove(listener);
    }

    @Override
    public void addCcmService(CcmService ccm) {
        if (ccm != null) {
            ccm.setNode(this);
            ccmList.add(ccm);
        }
    }

    @Override
    public void clearCcmServices() {
        for (CcmService sv : ccmList) {
            sv.setNode(null);
        }
        ccmList.clear();
    }

    @Override
    public void addProtocol(IUecsProtocolHandler protocol) {
        if (protocol != null) {
            protocols.add(protocol);
        }
    }

    @Override
    public void clearProtocols() {
        protocols.clear();
    }

    @Override
    public int countCcm() {
        return ccmList.size();
    }

    /*
     * イベントスレッドでタスク実行する.
     */
    void executeTask(Runnable task) {
        this.eventExecutor.execute(task);
    }

    @Override
    public InetAddress getBroadcastAddress() {
        return broadcastAddress;
    }

    @Override
    public int getCcmIndex(CcmService ccm) {
        return ccmList.indexOf(ccm);
    }

    @Override
    public CcmService getCcmService(int index) {
        if (index < 0 || index > ccmList.size() - 1) {
            return null;
        }
        return ccmList.get(index);
    }

    @Override
    public T getConfig() {
        return this.config;
    }

    @Override
    public InetAddress getIpAddress() {
        return ipAddress;
    }

    @Override
    public byte[] getMacAddress() {
        return macAddress;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getUecsID() {
        return uecsID;
    }

    @Override
    public String getUecsVersion() {
        return uecsVersion;
    }

    @Override
    public String getVender() {
        return vender;
    }

    @Override
    public int getStatus() {
        return status;
    }

    @Override
    public ActionMode getActionMode() {
        int modeBits = status & UecsConstants.STATUS_AREA_MODE;
        return ActionMode.getMode(modeBits);
    }

    @Override
    public void setActionMode(ActionMode mode) {
        int modeBits = status & UecsConstants.STATUS_AREA_MODE;
        status -= modeBits;
        status += mode.getBits();
    }

    @Override
    public boolean isActive() {
        return isActive;
    }

    @Override
    public List<CcmService> listCcmService() {
        return new ArrayList<CcmService>(ccmList);
    }

    @Override
    public CcmService removeCcmService(int index) {
        CcmService sv = ccmList.remove(index);
        if (sv != null) {
            sv.setNode(null);
        }
        return sv;
    }

    @Override
    public IDevice<?> getDevice(String deviceId) {
        return deviceMap.get(deviceId);
    }

    @Override
    public void addDevice(IDevice<?> device) {
        if (device != null) {
            deviceMap.put(device.getId(), device);
        }
    }

    @Override
    public void clearDevices() {
        deviceMap.clear();
    }

    @Override
    public List<IDevice<?>> listDevices() {
        return new ArrayList<IDevice<?>>(deviceMap.values());
    }

    @Override
    @SuppressWarnings("unchecked")
    public <U extends IDevice<?>> List<U> listDevices(Class<U> cls) {
        List<U> list = new ArrayList<U>();
        for (IDevice<?> compo : listDevices()) {
            if (cls.isInstance(compo)) {
                list.add((U) compo);
            }
        }
        return list;
    }

    @Override
    public void removeDevice(IDevice<?> device) {
        deviceMap.remove(device);
    }

    @Override
    public void sendPacket(InetAddress address, int port, byte[] data) {
        // ストップされた後に
        if (sendSocket == null) {
            return;
        }
        try {
            DatagramPacket packet = new DatagramPacket(data, data.length, address, port);
            sendSocket.send(packet);
        } catch (IOException e) {
            throw new NodeRuntimeException(e);
        }
    }

    /**
     * UDPブロードキャスト用アドレスを設定します。
     *
     * @param broadcastAddress ブロードキャストアドレス
     */
    public void setBroadcastAddress(InetAddress broadcastAddress) {
        this.broadcastAddress = broadcastAddress;
    }

    /**
     * ノードのIPアドレスを設定します。
     *
     * @param ipAddress IPアドレス
     */
    public void setIpAddress(InetAddress ipAddress) {
        this.ipAddress = ipAddress;
        if (ipAddress == null) {
            this.macAddress = null;
        } else {
            // MACアドレス設定
            NetworkInterface nic;
            try {
                nic = NetworkInterface.getByInetAddress(ipAddress);
                if (nic != null) {
                    macAddress = nic.getHardwareAddress();
                }
            } catch (SocketException e) {
                throw new NodeRuntimeException(e);
            }
        }
    }

    /**
     * ノード名称を設定します。
     *
     * @param name UECS実用通信規約に基づくASCIIコードの文字列
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * UECSIDを設定します。
     *
     * @param uecsID UECS研究会より認証された機種についてベンダーに発行されるID
     */
    public void setUecsID(String uecsID) {
        this.uecsID = uecsID;
    }

    /**
     * UECS仕様バージョンを設定します。
     *
     * @param uecsVersion バージョン表記文字列（"1.00-E10"等）
     */
    public void setUecsVersion(String uecsVersion) {
        this.uecsVersion = uecsVersion;
        for (CcmService sv : ccmList) {
            sv.getCcm().setUecsVersion(uecsVersion);
        }
    }

    /**
     * ベンダー名を設定します。
     *
     * @param vender ベンダー名称文字
     */
    public void setVender(String vender) {
        this.vender = vender;
    }

    @Override
    public void setStatus(int status) {
        this.status = status;
    }

    @Override
    public void onStatus(int onBits) {
        this.status |= onBits;
    }

    @Override
    public void offStatus(int offBits) {
        this.status &= ~offBits;
    }

    /**
     * 初期設定処理を行います。
     * デフォルト動作として以下の初期化を行います。
     * <ul>
     * <li>CcmServiceをすべてクリアします。</li>
     * <li>有効なNICを検索し、最初に見つかったIPを自IPとして設定します。</li>
     * <li>有効なNICが見つからなかった場合、localhostアドレスを設定します。</li>
     * <li>ブロードキャストアドレスを255.255.255.255に設定します。</li>
     * <li></li>
     * </ul>
     * 以下の場合はNodeRuntimeExceptionがスローされます。
     * <ul>
     * <li>ノード動作中の場合</li>
     * <li>IPアドレスやMACアドレスが設定できなかった場合</li>
     * </ul>
     *
     */
    @Override
    public void setup() {
        // ノード動作中は設定不可
        if (isActive()) {
            NodeRuntimeException e = new NodeRuntimeException("node is active.");
            notifyException(e);
            throw e;
        }

        // 内部データ初期化。
        clearCcmServices();
        clearDevices();

        setUecsVersion(config.getString(NodeConfig.KEY_UECS_VERSION, UecsConstants.DEFAULT_VERSION));
        setUecsID(config.getString(NodeConfig.KEY_UECS_ID, ""));
        setName(config.getString(NodeConfig.KEY_NODE_NAME, ""));
        setVender(config.getString(NodeConfig.KEY_VENDER, ""));

        // ノード動作状態CCMを登録
        CndCcm cndCcm = new CndCcm(this, config.getString(NodeConfig.KEY_NODE_TYPE, "cXX"));
        cndCcm.setRoom(config.getInt(NodeConfig.KEY_NODE_ROOM, 0));
        cndCcm.setRegion(config.getInt(NodeConfig.KEY_NODE_REGION, 0));
        cndCcm.setOrder(config.getInt(NodeConfig.KEY_NODE_ORDER, 0));
        cndCcm.setPriority(config.getInt(NodeConfig.KEY_NODE_PRIORITY, 0));
        cndCcmService = new CcmService(cndCcm);
        cndCcmService.setEnable(true);
        cndCcmService.setName(getName());
        cndCcmService.setValue(System.currentTimeMillis(), 0);
        addCcmService(cndCcmService);
        setStatus(UecsConstants.STATUS_NORMAL);

        try {
            // 自ホストIPを設定
            String ip = config.getString(NodeConfig.KEY_NODE_IP, "");
            InetAddress iadr = null;
            if (ip.length() > 0) {
                iadr = InetAddress.getByName(ip);
            } else {
                InetAddress tmp_adr = null;
                Enumeration<NetworkInterface> nics = NetworkInterface.getNetworkInterfaces();
                while (nics.hasMoreElements() && iadr == null) {
                    NetworkInterface nic = nics.nextElement();
                    Enumeration<InetAddress> adds = nic.getInetAddresses();
                    while (adds.hasMoreElements()) {
                        InetAddress adr = adds.nextElement();
                        if (adr instanceof Inet4Address && !adr.isLoopbackAddress()) {
                            tmp_adr = adr;
                            // 192.168.x.xを優先
                            if (adr.getHostAddress().startsWith("192.168.")) {
                                iadr = adr;
                                break;
                            }
                        }
                    }
                }
                // 192.168.x.x以外のIPしか見当たらなかったら、そちらを設定
                if (iadr == null && tmp_adr != null) {
                    iadr = tmp_adr;
                }
            }
            if (iadr == null) {
                // NICが存在しない場合に例外が発生するが、無視して空アドレスとする。
                try {
                    iadr = InetAddress.getLocalHost();
                } catch (Exception e) {

                }
            }
            setIpAddress(iadr);

            // ブロードキャストアドレス設定
            String bc = config.getString(NodeConfig.KEY_NODE_BROADCAST, "");
            if (bc.length() != 0) {
                InetAddress badr = InetAddress.getByName(bc);
                setBroadcastAddress(badr);
            } else {
                // デフォルトで設定する
                setBroadcastAddress(InetAddress.getByAddress(new byte[] {(byte) 0xFF, (byte) 0xFF, (byte) 0xFF,
                        (byte) 0xFF}));
            }

        } catch (Throwable t) {
            onStatus(UecsConstants.STATUS_ALART_INIT_FAILED);
            NodeRuntimeException  e = new NodeRuntimeException(t);
            notifyException(e);
            throw e;
        }
    }

    @Override
    public void start() {
        if (isActive) {
            return;
        }

        synchronized (this) {
            // 初期化できた安全なデバイス
            List<IDevice<?>> safeDevices = new ArrayList<IDevice<?>>();

            // デバイス初期化
            for (IDevice<?> device : deviceMap.values()) {
                try {
                    device.init(this);
                    safeDevices.add(device);
                } catch (Exception e) {
                    // １デバイス起動が失敗しても、全体が止まらないようにひとまず処理を進める。
                    onStatus(UecsConstants.STATUS_ALART_INIT_FAILED);
                    notifyException(e);
                }
            }

            // デバイス動作スタート
            for (IDevice<?> device : safeDevices) {
                try {
                    device.start();
                } catch (Exception e) {
                    // １デバイス起動が失敗しても、全体が止まらないようにひとまず処理を進める。
                    onStatus(UecsConstants.STATUS_ALART_INIT_FAILED);
                    notifyException(e);
                }
            }

            try {

                // イベント処理スレッド
                // TODO スレッド数はconfigから読むべきか？
                eventExecutor = Executors.newFixedThreadPool(3);

                // 送信ソケット
                sendSocket = new DatagramSocket();
                sendSocket.setBroadcast(true);

                // プロトコルサーバ起動
                for (IUecsProtocolHandler protocol : protocols) {
                    UdpServer server = new UdpServer(protocol);
                    server.start();
                    servers.add(server);
                }

                // 定周期実行
                cycleExecutor = Executors.newSingleThreadScheduledExecutor();
                int count = 0;
                for (CcmService ccmsv : ccmList) {
                    ccmsv.onStart(this);
                    long time = ccmsv.getCcm().getLevel().getCycleTime();
                    if (time > 0) {
                        // 送信間隔をずらして、短時間に送信処理が集中しないようにする
                        cycleExecutor.scheduleAtFixedRate(new CycleTask(ccmsv), config.getLong(DELAY_TIME, DEFAULT_DELAY)
                            * count++, time, TimeUnit.MILLISECONDS);
                    }
                }

                isActive = true;
            } catch (Throwable t) {
                onStatus(UecsConstants.STATUS_ALART_INIT_FAILED);
                NodeRuntimeException  e = new NodeRuntimeException(t);
                notifyException(e);
                throw e;
            }

        }
        for(IUecsNodeListener listener : listeners) {
            listener.nodeStarted(this);
        }
    }

    @Override
    public void stop() {
        if (!isActive) {
            return;
        }
        // 最初に不活性フラグセット
        isActive = false;
        // 例外発生はまとめる
        List<Throwable> exceptions = new ArrayList<Throwable>();

        synchronized (this) {

            // CCM受信サービス停止
            for (UdpServer sv : servers) {
                try {
                    sv.stop();
                } catch (Exception e) {
                    exceptions.add(e);
                }
            }
            servers.clear();

            // 受信イベントスレッド停止
            try {
                eventExecutor.shutdown();
                eventExecutor = null;
            } catch (Throwable t) {
                exceptions.add(t);
            }

            // CCMサービス停止
            for (CcmService ccmsv : ccmList) {
                try {
                    ccmsv.onStop(this);
                } catch (Exception e) {
                    exceptions.add(e);
                }
            }

            // CCM周期動作スレッド停止
            try {
                cycleExecutor.shutdown();
                cycleExecutor = null;
            } catch (Throwable t) {
                exceptions.add(t);
            }

            // UDP-CCM送信ソケットを閉じる
            try {
                if (sendSocket != null && !sendSocket.isClosed()) {
                    sendSocket.close();
                    sendSocket = null;
                }
            } catch (Throwable t) {
                exceptions.add(t);
            }

        }

        // デバイス終了処理でノードにアクセスされることもありえるので、同期ブロックを分ける
        synchronized (this) {
            // デバイス動作ストップ
            for (IDevice<?> device : deviceMap.values()) {
                try {
                    device.stop();
                } catch (Throwable t) {
                    exceptions.add(t);
                }
            }
        }

        if (exceptions.size() > 0) {
            for (Throwable t : exceptions) {
                notifyException(new NodeRuntimeException(t));
            }
        }

        for(IUecsNodeListener listener : listeners) {
            listener.nodeStopped(this);
        }
    }

    /**
     * ノード動作中に例外が発生した場合にハンドラメソッドとして起動されます。
     * 内部でノードリスナへの通知が行われます。
     * @param e 例外
     */
    protected void notifyException(Exception e) {
        for(IUecsNodeListener listener : listeners) {
            listener.handleNodeException(this, e);
        }
    }

}
