/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.protocol.handler;

import com.wabit.uecs.Ccm;
import com.wabit.uecs.CcmLevel;
import com.wabit.uecs.CcmService;
import com.wabit.uecs.IUecsNode;
import com.wabit.uecs.UecsConstants;
import com.wabit.uecs.UecsRequest;
import com.wabit.uecs.UecsResponse;
import com.wabit.uecs.protocol.DataCcm;
import com.wabit.uecs.protocol.RequestCcm;
import com.wabit.uecs.protocol.XmlCcm;

/**
 * データポート処理を実行するハンドラーです。
 * UECS通信規約の図3-1「データ送信CCM受信時の、受信ノードの挙動に関するフローチャート」に従います。<br>
 * UECS通信規約の表3-5「データ要求CCM」の補足内容の処理に従います。<br>
 * <ol>
 * <li>送信タイミングがレベルB</li>
 * <li>type属性が等しく</li>
 * <li>自身に関連する room, region, order の各属性値(全系統向けの0は関係がある)</li>
 * </ol>
 * @author WaBit
 *
 */
public class DataProtocolHandler extends AbstractProtocolHandler {

    public DataProtocolHandler(IUecsNode<?> node) {
        super(node);
    }

    @Override
    public int getPort() {
        return UecsConstants.DATA_PORT;
    }

    @Override
    protected void handleCcm(UecsRequest req, UecsResponse res, XmlCcm ccm) throws Exception {
        if (ccm instanceof DataCcm) {
            handleDataCcm(req, res, (DataCcm)ccm);
        } else if (ccm instanceof RequestCcm) {
            handleRequestCcm(req, res, (RequestCcm)ccm);
        } else {
            throw new Exception("illegal packet : " + ccm.toString());
        }
    }


    /**
     * REQUEST CCM受信処理です。
     * @param req リクエスト
     * @param res レスポンス
     * @param rCcm リクエストCCM
     */
    public void handleRequestCcm(UecsRequest req, UecsResponse res, RequestCcm rCcm) {
        for (CcmService sv : getNode().listCcmService()) {
            // サービスが有効かつ送信側のみ処理を行う
            if (sv.isEnable() && sv.getCcm().getSide() == UecsConstants.SENDER) {
                processResponse(req, res, rCcm, sv);
            }
        }
    }

    // DATA CCM受信処理
    private void handleDataCcm(UecsRequest req, UecsResponse res, DataCcm rCcm) {
        for (CcmService sv : getNode().listCcmService()) {
            // CCMが有効かつ受信側のみ処理を行う
            if (sv.isEnable() && sv.getCcm().getSide() == UecsConstants.RECEIVER) {
                processReceive(sv, rCcm);
            }
        }
    }

    /*
     * リクエスト応答処理
     */
    private void processResponse(UecsRequest req, UecsResponse res, RequestCcm rCcm, CcmService sv) {
        Ccm ccm = sv.getCcm();
        CcmLevel level = ccm.getLevel();

        if ((level.getCategory() == 'B')
                && ccm.getType().equals(rCcm.getType())
                && (ccm.getRoom() == rCcm.getRoom()
                        || rCcm.getRoom() == 0)
                && (ccm.getRegion() == rCcm.getRegion()
                        || rCcm.getRegion() == 0)
                && (ccm.getOrder() == rCcm.getOrder()
                        || rCcm.getOrder() == 0)) {

            String data = sv.getCcm().getStringValue();
            if (data != null && data.length() != 0) {
                DataCcm dccm = createDataCcm(ccm);

                res.setAddress(getNode().getBroadcastAddress());
                res.setPort(dccm.getPort());
                res.setData(dccm.toXmlBytes());
            }
        }
    }
    /*
     * 受信時処理を実行します。
     */
    private void processReceive(CcmService ccmSv, DataCcm rCcm) {
        Ccm bCcm = ccmSv.getCcm();
        DataCcm vCcm = (DataCcm)ccmSv.getAcceptCcm();
        // 初回受信時に優先順位の最も低い属性値をvCcmに初期値として登録
        if (vCcm == null) {
            vCcm = createDataCcm(bCcm);
            vCcm.setRoom(0);
            vCcm.setRegion(0);
            vCcm.setOrder(0);
            vCcm.setPriority(30);
        }

        boolean change = false;
        if (bCcm.getType().equals(rCcm.getType())
                && (bCcm.getRoom() == rCcm.getRoom()
                        || rCcm.getRoom() == 0)
                && (bCcm.getRegion() == rCcm.getRegion()
                        || rCcm.getRegion() == 0)
                && (bCcm.getOrder() == rCcm.getOrder()
                        || rCcm.getOrder() == 0)) {

            CcmLevel level = bCcm.getLevel();
            // bCCMの登録がレベルAかS && vCCMの有効時間内
            if ((level.getCategory() == 'A' || level.getCategory() == 'S')
                    && ((System.currentTimeMillis() - vCcm
                            .getReceivedTime()) <= level.getAvailableTime())) {
                // vCCM.priority = rCCM.priority
                if (vCcm.getPriority() == rCcm.getPriority()) {
                    // vCCMとrCCMのDATAタグ内の属性値がすべて等しい
                    if (vCcm.getRoom() == rCcm.getRoom()
                            && vCcm.getRegion() == rCcm.getRegion()
                            && vCcm.getOrder() == rCcm.getOrder()
                            && vCcm.getPriority() == rCcm.getPriority()) {
                        // IPアドレスが小さい方または同じものをvCCMとする
                        if (vCcm.getIpAddress() == null || rCcm.getIpAddress() == null) {
                            // 旧バージョン対応
                            change = true;
                        } else {
                            byte[] vip = vCcm.getIpAddress().getAddress();
                            byte[] rip = rCcm.getIpAddress().getAddress();
                            for (int i = 0; i < vip.length; i++) {
                                // vCCMの方が小さいときは変更なし
                                if (vip[i] < rip[i]) {
                                    change = false;
                                    break;
                                } else if (vip[i] > rip[i]) {
                                    change = true;
                                    break;
                                } else {
                                    change = true;
                                }
                            }
                        }
                    }
                    // 基本規約第4-2(基)表の優先順位によってrCCMをvCCMとするかを決定
                    else if (calcPriority(bCcm, vCcm) > calcPriority(bCcm, rCcm)) {
                        change = true;
                    }
                }
                // priorityの低い方をvCCMとする
                else if (vCcm.getPriority() > rCcm.getPriority()) {
                    change = true;
                }
            } else {
                change = true;
            }
        }
        // CCMを入れ替える
        if (change) {
            ccmSv.setAcceptCcm(rCcm);
        }

    }

    /*
     * 基本規約第4-2(基)表の優先順位計算を行います。
     * @param bCcm
     * @param ccm
     * @return 優先度(低い方が上位)
     */
    private int calcPriority(Ccm bCcm, Ccm ccm) {
        // ビット演算ロジックで優先度を決定
        int grade = 0;
        // room(100)
        if (bCcm.getRoom() == ccm.getRoom()) {
            grade += 4;
        }
        // region(010)
        if (bCcm.getRegion() == ccm.getRegion()) {
            grade += 2;
        }
        // order(001)
        if (bCcm.getOrder() == ccm.getOrder()) {
            grade += 1;
        }
        return 8 - grade;
    }

    // CCM作成
    private DataCcm createDataCcm(Ccm ccm) {
        DataCcm dccm = new DataCcm();
        dccm.setUecsVersion(ccm.getUecsVersion());
        dccm.setType(ccm.getType());
        dccm.setRoom(ccm.getRoom());
        dccm.setRegion(ccm.getRegion());
        dccm.setOrder(ccm.getOrder());
        dccm.setPriority(ccm.getPriority());
        dccm.setIpAddress(getNode().getIpAddress());
        dccm.setCast(ccm.getCast());
        dccm.setUnit(ccm.getUnit());
        dccm.setValue(ccm.getNumberValue());
        return dccm;
    }


}