/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.device.actuator;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.wabit.uecs.ActionMode;
import com.wabit.uecs.Ccm;
import com.wabit.uecs.CcmService;
import com.wabit.uecs.ICcmServiceListener;
import com.wabit.uecs.IUecsNode;
import com.wabit.uecs.NodeConfig;
import com.wabit.uecs.UecsConstants;
import com.wabit.uecs.ccm.OprCcm;
import com.wabit.uecs.ccm.RcACcm;
import com.wabit.uecs.ccm.RcMCcm;
import com.wabit.uecs.device.AbstractComponent;
import com.wabit.uecs.device.ComponentConfig;
import com.wabit.uecs.device.IDevice;

/**
 * アクチュエータ部品を表現するクラスです。
 *
 * @author WaBit
 */
public abstract class ActuatorComponent<T extends ComponentConfig> extends AbstractComponent<T> {

    private ActionMode actionMode = ActionMode.Autonomy;
    private Map<ActionMode, IActuatorAction> modeActions = new HashMap<ActionMode, IActuatorAction>();
    private List<CcmService> oprServices = new ArrayList<CcmService>();
    private List<CcmService> rcAServices = new ArrayList<CcmService>();
    private List<CcmService> rcMServices = new ArrayList<CcmService>();

    // 複数系統での代表値動作用
    boolean isRepresentative;
    private ICcmServiceListener ccmListener = new InnerCcmListener();

    /**
     * コンストラクタ.
     * @param id コンポーネントID
     * @param config 設定値オブジェクト
     */
    public ActuatorComponent(String id, T config) {
        super(id, config);
        // 値更新用リスナ登録
    }

    /**
     * opr CCMサービス一覧を取得します。
     * @return 優先順位準のCCMリスト
     */
    public CcmService[] getOprServices() {
        return oprServices.toArray(new CcmService[oprServices.size()]);
    }

    /**
     * rcA CCMサービス一覧を取得します。
     * @return 優先順位準のCCMリスト
     */
    public CcmService[] getRcAServices() {
        return rcAServices.toArray(new CcmService[rcAServices.size()]);
    }

    /**
     * rcM CCMサービス一覧を取得します。
     * @return 優先順位準のCCMリスト
     */
    public CcmService[] getRcMServices() {
        return rcMServices.toArray(new CcmService[rcMServices.size()]);
    }

    /**
     * 値を設定します。
     * 同時に連動するCCMの値も変更されます。
     */
    @Override
    public void setValue(Number value) {
        super.setValue(value);
        // 値変更時に状態通知CCM(opr.kNN)の値を変更
        for (CcmService oprService : oprServices) {
            oprService.updateValue(value);
        }
    }

    /**
     * 値が有効期限内で最も優先順位の高いrcAサービスを取得します。
     * @return CCMサービス
     */
    public CcmService getActiveRcAService() {
        for (CcmService rcASv : rcAServices) {
            if (!rcASv.isExpired()) {
                return rcASv;
            }
        }
        return null;
    }

    /**
     * 値が有効期限内で最も優先順位の高いrcMサービスを取得します。
     * @return CCMサービス
     */
    public CcmService getActiveRcMService() {
        for (CcmService rcMSv : rcMServices) {
            if (!rcMSv.isExpired()) {
                return rcMSv;
            }
        }
        return null;
    }

    /**
     * 現在の動作モードを取得します。
     * @return 動作モード
     */
    public ActionMode getActionMode() {
        return actionMode;
    }

    /**
     * 強制的にモード値を設定します。
     * 主に初期化処理時に継承クラスから利用することを想定しています。
     * @param mode 動作モード
     */
    protected void setActionMode(ActionMode mode) {
        this.actionMode = mode;
        if (getDevice() != null && getDevice().getNode() != null) {
            IUecsNode<?> node = getDevice().getNode();
            if (isRepresentative()) {
                node.setActionMode(this.actionMode);
            }
        }
    }

    /**
     * 動作モードを切り替えます。
     * @param mode 動作モード
     */
    public synchronized void changeMode(ActionMode mode) throws Exception {
        IActuatorAction action = null;

        // 現在のアクションをストップして新しいアクションを起動する
        if (getActionMode() != mode) {
            action = modeActions.get(getActionMode());
            if (action != null) {
                action.onStop(this);
            }
            action = modeActions.get(mode);
            if (action != null) {
                action.onStart(this);
            }
        }
        setActionMode(mode);
    }

    /**
     * 動作モードに対応するアクションを設定します。
     * @param mode 動作モード
     * @param action 動作モードに連動するアクション
     */
    public void setAction(ActionMode mode, IActuatorAction action) {
        modeActions.put(mode, action);
        if (action instanceof ICcmListenerAction) {
            ICcmListenerAction ccmAction = (ICcmListenerAction) action;
            for (CcmService ccmSv : ccmAction.getCcmServices()) {
                if (ccmSv != null) {
                    ccmSv.addListener(ccmListener);
                }
            }
        }
    }

    /**
     * モードに設定されたアクションを取得します。
     * @param mode 動作モード
     * @return アクション
     */
    public IActuatorAction getAction(ActionMode mode) {
        return modeActions.get(mode);
    }

    /**
     * ノード代表値であるかを返します。
     * @return 代表値であればtrue
     */
    public boolean isRepresentative() {
        return isRepresentative;
    }

    @Override
    public void notifyException(Exception e) {
        super.notifyException(e);
    }

    /*
     * (non-Javadoc)
     * @see com.wabit.uecs.device.IComponent#init(com.wabit.uecs.device.IDevice)
     */
    @Override
    public void init(IDevice<?> device) throws Exception {
        setDevice(device);

        // 対応するCCMを登録する
        IUecsNode<?> node = getDevice().getNode();
        NodeConfig nodeConfig = node.getConfig();
        String nodeType = nodeConfig.getString(NodeConfig.KEY_NODE_TYPE);
        int nodeRoom = nodeConfig.getInt(NodeConfig.KEY_NODE_ROOM, 0);
        int nodeRegion = nodeConfig.getInt(NodeConfig.KEY_NODE_REGION, 0);
        int nodeOrder = nodeConfig.getInt(NodeConfig.KEY_NODE_ORDER, 0);
        int nodePriority = nodeConfig.getInt(NodeConfig.KEY_NODE_PRIORITY, 0);

        ComponentConfig config = getConfig();
        String name = config.getString(ComponentConfig.KEY_COMPONENT_NAME);
        String ccmInfoName = config.getString(ComponentConfig.KEY_CCM_INFO_NAME);
        String ccmNodeType = config.getString(ComponentConfig.KEY_CCM_NODE_TYPE);
        String unit = config.getString(ComponentConfig.KEY_CCM_UNIT);
        int ccmRoom = config.getInt(ComponentConfig.KEY_CCM_ROOM, 0);
        int ccmRegion = config.getInt(ComponentConfig.KEY_CCM_REGION, 0);
        int ccmOrder = config.getInt(ComponentConfig.KEY_CCM_ORDER, 0);
        int ccmPriority = config.getInt(ComponentConfig.KEY_CCM_PRIORITY, 0);
        int ccmLine = config.getInt(ComponentConfig.KEY_CCM_LINE, 0);

        // ノード代表値としてあつかうかを判断する
        isRepresentative = (config.getString(ComponentConfig.KEY_CCM_INFO_NAME, "").length() == 0
                && config.getInt(ComponentConfig.KEY_CCM_LINE, 0) == 0
                && nodeType.equals(config.getString(ComponentConfig.KEY_CCM_NODE_TYPE))
                && nodeRoom == ccmRoom && nodeRegion == ccmRegion
                && nodeOrder == ccmOrder && nodePriority == ccmPriority
                );

        // 別名が付けられているが、ノード代表値として明示された場合は、新たにCCM登録する
        if (!isRepresentative && config.getBoolean(ComponentConfig.KEY_IS_REPRESENTATIVE, false)) {
            isRepresentative = true;

            OprCcm opr = new OprCcm(nodeType);
            opr.setSide(UecsConstants.SENDER);
            opr.setUnit(unit);
            opr.setRoom(nodeRoom);
            opr.setRegion(nodeRegion);
            opr.setOrder(nodeOrder);
            opr.setPriority(nodePriority);
            CcmService oprServiceR = new CcmService(opr);
            oprServiceR.setName(name);
            node.addCcmService(oprServiceR);
            oprServices.add(oprServiceR);

            RcACcm rcA = new RcACcm(nodeType);
            rcA.setSide(UecsConstants.RECEIVER);
            rcA.setUnit(unit);
            rcA.setRoom(nodeRoom);
            rcA.setRegion(nodeRegion);
            rcA.setOrder(nodeOrder);
            rcA.setPriority(nodePriority);
            CcmService rcAServiceR = new CcmService(rcA);
            rcAServiceR.setName(name);
            rcAServiceR.addListener(ccmListener);
            node.addCcmService(rcAServiceR);
            rcAServices.add(rcAServiceR);

            RcMCcm rcM = new RcMCcm(nodeType);
            rcM.setSide(UecsConstants.RECEIVER);
            rcM.setUnit(unit);
            rcM.setRoom(nodeRoom);
            rcM.setRegion(nodeRegion);
            rcM.setOrder(nodeOrder);
            rcM.setPriority(nodePriority);
            CcmService rcMServiceR = new CcmService(rcM);
            rcMServiceR.setName(name);
            rcMServiceR.addListener(ccmListener);
            node.addCcmService(rcMServiceR);
            rcMServices.add(rcMServiceR);

        }

        OprCcm opr = new OprCcm(ccmNodeType, ccmInfoName, ccmLine);
        opr.setSide(UecsConstants.SENDER);
        opr.setUnit(unit);
        opr.setRoom(ccmRoom);
        opr.setRegion(ccmRegion);
        opr.setOrder(ccmOrder);
        opr.setPriority(ccmPriority);
        CcmService oprService = new CcmService(opr);
        oprService.setName(name);
        node.addCcmService(oprService);
        oprServices.add(oprService);

        RcACcm rcA = new RcACcm(ccmNodeType, ccmInfoName, ccmLine);
        rcA.setSide(UecsConstants.RECEIVER);
        rcA.setUnit(unit);
        rcA.setRoom(ccmRoom);
        rcA.setRegion(ccmRegion);
        rcA.setOrder(ccmOrder);
        rcA.setPriority(ccmPriority);
        CcmService rcAService = new CcmService(rcA);
        rcAService.setName(name);
        rcAService.addListener(ccmListener);
        node.addCcmService(rcAService);
        rcAServices.add(rcAService);

        RcMCcm rcM = new RcMCcm(ccmNodeType, ccmInfoName, ccmLine);
        rcM.setSide(UecsConstants.RECEIVER);
        rcM.setUnit(unit);
        rcM.setRoom(ccmRoom);
        rcM.setRegion(ccmRegion);
        rcM.setOrder(ccmOrder);
        rcM.setPriority(ccmPriority);
        CcmService rcMService = new CcmService(rcM);
        rcMService.setName(name);
        rcMService.addListener(ccmListener);
        node.addCcmService(rcMService);
        rcMServices.add(rcMService);

        super.init(device);
    }

    /*
     * (non-Javadoc)
     * @see com.wabit.uecs.device.IComponent#start()
     */
    @Override
    public void start() throws Exception {
        onStart();
        // 初期処理
        IActuatorAction action = modeActions.get(actionMode);
        if (action != null) {
            action.onStart(this);
        }
        notifyStarted();
    }

    /*
     * (non-Javadoc)
     * @see com.wabit.uecs.device.IComponent#stop()
     */
    @Override
    public void stop() throws Exception {
        // 終了処理
        IActuatorAction action = modeActions.get(actionMode);
        if (action != null) {
            action.onStop(this);
        }
        onStop();
        notifyStopped();
    }

    // 共通処理
    private boolean acceptCcmAction(ActionMode mode, Ccm ccm) throws Exception {
        IActuatorAction action = modeActions.get(mode);
        if (action != null && action instanceof ICcmListenerAction && ((ICcmListenerAction) action).isAccepted(ccm)) {
            return true;
        }
        return false;
    }

    // 共通処理
    private boolean isInactive(ActionMode mode) throws Exception {
        IActuatorAction action = modeActions.get(mode);
        if (action == null
                || (action instanceof ICcmListenerAction && ((ICcmListenerAction) action).isExpired())) {
            return true;
        }
        return false;
    }


    /**
     * 関連付けられたCCM(rcA, rcM, センサー連動用CCM等)が更新された際の動作が実装されています。
     *
     */
    private class InnerCcmListener implements ICcmServiceListener {

        @Override
        public synchronized void ccmValueChanged(CcmService service, Ccm ccm) {

            // 無効状態判定
            if (ActionMode.Outage == getActionMode()) {
                return;
            }

            try {
                ActionMode exeMode = null;
                Number num = ccm.getNumberValue();
                // 低い優先度順にチェック。
                if (ActionMode.Standalone.isPrior(getActionMode())
                        && acceptCcmAction(ActionMode.Standalone, ccm)) {
                    exeMode = ActionMode.Standalone;
                } else if (ActionMode.Autonomy.isPrior(getActionMode())
                        && acceptCcmAction(ActionMode.Autonomy, ccm)) {
                    exeMode = ActionMode.Autonomy;
                } else if (ActionMode.rcA.isPrior(getActionMode()) && service == getActiveRcAService()) {
                    if (num == null || num.intValue() < -100 || num.intValue() > 100) {
                        // 通信実用規約1.00-E10に従い、範囲外値はより低い優先順位の動作指示に移行
                        if (getAction(ActionMode.Autonomy) != null) {
                            exeMode = ActionMode.Autonomy;
                        } else {
                            exeMode = ActionMode.Standalone;
                        }
                    } else {
                        exeMode = ActionMode.rcA;
                    }
                } else if (ActionMode.WEB.isPrior(getActionMode())
                        && acceptCcmAction(ActionMode.WEB, ccm)) {
                    exeMode = ActionMode.WEB;
                } else if (ActionMode.rcM.isPrior(getActionMode()) && service == getActiveRcMService()) {
                    if (num == null || num.intValue() < -100 || num.intValue() > 100) {
                        // 通信実用規約1.00-E10に従い、範囲外値はより低い優先順位の動作指示に移行
                        if (getAction(ActionMode.WEB) != null) {
                            exeMode = ActionMode.WEB;
                        } else  {
                            if (getActiveRcAService() != null) {
                                exeMode = ActionMode.rcA;
                            } else {
                                exeMode = ActionMode.Standalone;
                            }
                        }
                    } else {
                        exeMode = ActionMode.rcM;
                    }
                } else if (ActionMode.Interlock.isPrior(getActionMode())
                        && acceptCcmAction(ActionMode.Interlock, ccm)) {
                    exeMode = ActionMode.Interlock;
                }
                if (exeMode != null) {
                    changeMode(exeMode);
                    IActuatorAction action = getAction(exeMode);
                    if (action != null) {
                        action.execute(ActuatorComponent.this);
                    }
                }
            } catch (Exception e) {
                // アクションが異常終了した場合、強制停止モードに変更する
                try {
                    changeMode(ActionMode.Outage);
                } catch (Exception e2) {
                    notifyException(e2);
                }
                notifyException(e);
            }

        }

        /*
         * 関連付けられたCCM(rcA, rcM, センサー連動用CCM等)が期限切れした際の動作が実装されています。
         */
        @Override
        public synchronized void ccmExpired(CcmService service, Ccm ccm) {

            try {
                ActionMode exeMode = null;
                // モードを移行する
                switch (getActionMode()) {
                case Outage:
                    break;
                case Interlock:
                    if (isInactive(ActionMode.Interlock)) {
                        exeMode = ActionMode.rcM;
                    } else {
                        break;
                    }
                case rcM:
                    if (getActiveRcMService() == null) {
                        exeMode = ActionMode.WEB;
                    } else {
                        break;
                    }
                case WEB:
                    if (isInactive(ActionMode.WEB)) {
                        exeMode = ActionMode.rcA;
                    } else {
                        break;
                    }
                case rcA:
                    if (getActiveRcAService() == null) {
                        exeMode = ActionMode.Autonomy;
                    } else {
                        break;
                    }
                case Autonomy:
                    if (isInactive(ActionMode.Autonomy)) {
                        exeMode = ActionMode.Standalone;
                    } else {
                        break;
                    }
                case Standalone:
                    break;
                }

                if (exeMode != null) {
                    changeMode(exeMode);
                    IActuatorAction action = getAction(exeMode);
                    if (action != null) {
                        action.execute(ActuatorComponent.this);
                    }
                }
            } catch (Exception e) {

                // アクションが異常終了した場合、強制停止モードに変更する
                actionMode = ActionMode.Outage;
                if (modeActions.get(ActionMode.Outage) != null) {
                    try {
                        modeActions.get(ActionMode.Outage).execute(ActuatorComponent.this);
                    } catch (Exception e2) {
                        notifyException(e2);
                    }
                }
                notifyException(e);
            }
        }

        @Override
        public void ccmSent(CcmService source, Ccm value) {
            // 何もしない
        }

        @Override
        public void ccmReceived(CcmService source, Ccm value) {
            // 何もしない
        }

    }

}
