/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.device;

import java.util.List;

import com.wabit.uecs.IUecsNode;


/**
 * デバイスインタフェースです。
 * センサーやアクチュエータのコンポーネントを内包した機器を表現します。
 * @author WaBit
 *
 * @param <T> 関連付けるデバイス設定クラス
 */
public interface IDevice<T extends DeviceConfig> {

	/**
	 * ユニークIDを取得します。
	 * @return ID
	 */
	public String getId();

	/**
	 * 設定オブジェクトを取得します。
	 * @return 設定オブジェクト
	 */
	public T getConfig();

	/**
     * 所属するノードインスタンスを取得します。
     * @return ノードインスタンス
     */
    public IUecsNode<?> getNode();

    /**
     * リスナーを登録します。
     * @param listener リスナー
     */
    public void addListener(IDeviceListener listener);

    /**
     * リスナー一覧を取得します。
     * @return リスナー一覧
     */
    public List<IDeviceListener> getListeners();

    /**
     * リスナーを削除します。
     * @param listener 削除したいリスナー
     * @return 削除されるとtrue
     */
    public boolean removeListener(IDeviceListener listener);

    /**
     * 保有しているコンポーネントを取得します。
     * @param componentId コンポーネントID
     * @return IDがマッチしたコンポーネント。無い場合はnull。
     */
    public IComponent<?> getComponent(String componentId);

    /**
     * コンポーネントを登録します。
     * IDが重複する場合は、入れ替わります。
     * @param component コンポーネントインスタンス
     */
    public void addComponent(IComponent<?> component);

    /**
     * コンポーネントリストを取得します。
     * @return コンポーネント一覧
     */
    public List<IComponent<?>> listComponents();

    /**
     * 型でフィルタ指定してコンポーネントリストを取得します。
     * @param cls フィルタする型
     * @return 型がマッチしたコンポーネント一覧
     */
	public <U extends IComponent<?>> List<U> listComponents(Class<U> cls);

    /**
     * コンポーネントを削除します。
     * @param componentId 削除したいコンポーネントのID
     * @return 削除されたコンポーネントインスタンス。IDが一致しない場合はnull。
     */
    public IComponent<?> removeComponent(String componentId);

    /**
     * 初期化処理を実行します。
     * @param node 所属するノード
     * @throws DeviceException 処理に失敗するとスローされます。
     */
    public void init(IUecsNode<?> node) throws DeviceException;

    /**
     * デバイス動作を開始します。
     * @throws Exception 処理に失敗するとスローされます。
     */
    public void start() throws DeviceException;

    /**
     * デバイス動作を終了させます。
     * @throws DeviceException 処理に失敗するとスローされます。
     */
    public void stop() throws DeviceException;

}
