/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs;


/**
 * CCMの通信レベルクラスです。
 *
 * @author WaBit
 *
 */
public class CcmLevel {

    /** UECS仕様書で定義されている送信レベル : A-1S-0 */
    public static final CcmLevel A_1S_0 = new CcmLevel("A-1S-0");
    /** UECS仕様書で定義されている送信レベル : A-1S-1 */
    public static final CcmLevel A_1S_1 = new CcmLevel("A-1S-1");
    /** UECS仕様書で定義されている送信レベル : A-10S-0 */
    public static final CcmLevel A_10S_0 = new CcmLevel("A-10S-0");
    /** UECS仕様書で定義されている送信レベル : A-10S-1 */
    public static final CcmLevel A_10S_1 = new CcmLevel("A-10S-1");
    /** UECS仕様書で定義されている送信レベル : A-1M-0 */
    public static final CcmLevel A_1M_0 = new CcmLevel("A-1M-0");
    /** UECS仕様書で定義されている送信レベル : A-1M-1 */
    public static final CcmLevel A_1M_1 = new CcmLevel("A-1M-1");
    /** UECS仕様書で定義されている送信レベル : B-0 */
    public static final CcmLevel B_0 = new CcmLevel("B-0");
    /** UECS仕様書で定義されている送信レベル : B-1 */
    public static final CcmLevel B_1 = new CcmLevel("B-1");
    /** UECS仕様書で定義されている送信レベル : S-1S-0 */
    public static final CcmLevel S_1S_0 = new CcmLevel("S-1S-0");
    /** UECS仕様書で定義されている送信レベル : S-1M-0 */
    public static final CcmLevel S_1M_0 = new CcmLevel("S-1M-0");

    private String code;
    private char category;
    private long cycleTime;
    private long availableTime;
    private boolean isAck;

    /**
     * コンストラクタ.
     *
     * @param code レベル表記文字列
     */
    public CcmLevel(String code) {
        this.code = code;
        try {
            parseCode(code);
        } catch (Throwable t) {
            throw new NodeRuntimeException("LV spec violation : " + code);
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o instanceof CcmLevel) {
            return ((CcmLevel) o).code.equals(this.code);
        }
        return false;
    }

    /**
     * 有効時間を取得します。
     *
     * @return 有効時間(msec)
     */
    public long getAvailableTime() {
        return availableTime;
    }

    /**
     * レベルカテゴリ文字を取得します。
     *
     * @return カテゴリ文字 ('A', 'B', 'S' etc...)
     */
    public char getCategory() {
        return category;
    }

    /**
     * 周期時間を取得します。
     *
     * @return 周期時間(msec)
     */
    public long getCycleTime() {
        return cycleTime;
    }

    @Override
    public int hashCode() {
        return code.hashCode();
    }

    /**
     * 値変更通知フラグを取得します。
     *
     * @return 通知する場合はtrue
     */
    public boolean isAck() {
        return isAck;
    }

    private void parseCode(String code) {
        String[] vals = code.split("-");
        if (vals.length > 1 && vals[0].length() == 1) {
            category = vals[0].charAt(0);
            if (vals[1].endsWith("S")) {
                cycleTime = Integer.parseInt(vals[1].substring(0,
                        vals[1].length() - 1)) * 1000L;
            } else if (vals[1].endsWith("M")) {
                cycleTime = Integer.parseInt(vals[1].substring(0,
                        vals[1].length() - 1)) * 60L * 1000L;
            }
            availableTime = cycleTime * 3L;
        }
        isAck = "1".equals(vals[vals.length - 1]);
    }

    @Override
    public String toString() {
        return code;
    }
}
