/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.protocol;

import java.io.CharArrayWriter;
import java.io.StringReader;
import java.io.UnsupportedEncodingException;

import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

import com.wabit.uecs.UecsConstants;

/**
 * SAX実装によるXML形式のCCM解析クラスです。
 *
 * @author WaBit
 *
 */
public class XmlCcmParser extends DefaultHandler {

    /** SAXパーサー生成用 */
    private static SAXParserFactory spfactory = SAXParserFactory.newInstance();

    /** UECSタグ発見フラグ */
    private boolean isUecs;

    /** UECSバージョン */
    private String version;

    /** 読込中CCM */
    private XmlCcm xmlCcm;

    /** タグ値読込用バッファー */
    private CharArrayWriter buffer;

    /**
     * XMLバイト列を解析して、CCMを作成します。
     *
     * @param xmlBytes XMLバイナリ配列
     * @return 解析されたCCM
     * @throws CcmParserException 解析に失敗するとスローされます。
     */
    public synchronized XmlCcm parseBytes(byte[] xmlBytes) throws CcmParserException {
        // 長さチェック
        if (xmlBytes == null || xmlBytes.length == 0) {
            return null;
        }

        String ccm = null;
        try {
            ccm = new String(xmlBytes, UecsConstants.DEFAULT_CHARSET);
            return parseXml(ccm);
        } catch (UnsupportedEncodingException e) {
            // ありえないはず.
            throw new CcmParserException(e.getMessage(), e, ccm);
        }

    }

    /**
     * XML文字列を解析して、CCMを作成します。
     * @param xml 解析対象文字列
     * @return 解析されたCCM
     * @throws CcmParserException 解析に失敗するとスローされます。
     */
    public synchronized XmlCcm parseXml(String xml) throws CcmParserException {
        // 長さチェック
        if (xml == null || xml.length() == 0) {
            return null;
        }

        // SAXパーサーを生成
        SAXParser saxParser;
        try {
            saxParser = spfactory.newSAXParser();
        } catch (Exception e) {
            // 通常起こりえないはず
            throw new RuntimeException(e);
        }

        InputSource source = new InputSource(new StringReader(xml));
        try {
            saxParser.parse(source, this);
        } catch (Exception e) {
            throw new CcmParserException("CCM parse error.", e, xml);
        }
        if (xmlCcm == null) {
            throw new CcmParserException("unknown CCM.", null, xml);
        }

        return xmlCcm;

    }

    /*
     * (非 Javadoc)
     * @see org.xml.sax.helpers.DefaultHandler#startDocument()
     */
    @Override
    public void startDocument() {
        xmlCcm = null;
        version = null;
        buffer = new CharArrayWriter();
    }

    /*
     * (非 Javadoc)
     * @see org.xml.sax.helpers.DefaultHandler#startElement(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
     */
    @Override
    public void startElement(String uri, String localName, String qName,
            Attributes attributes) throws SAXException {

        // UECSタグで始まるかどうかの判定
        if (xmlCcm == null && XmlCcm.TAG_UECS.equals(qName)) {
            isUecs = true;
            version = attributes.getValue(XmlCcm.ATTR_VER);
            return;
        }

        // UECSメッセージでない場合は解析を行わない
        if (!isUecs) {
            return;
        }

        // CCMの種別判定
        if (xmlCcm == null) {
            xmlCcm = createCcmCcm(qName);
            xmlCcm.setUecsVersion(version);
        }

        // 属性値設定
        try {
            xmlCcm.setXmlAttributes(qName, attributes);
        } catch (Exception e) {
            throw new SAXException(e);
        }
    }

    /*
     * (非 Javadoc)
     * @see org.xml.sax.helpers.DefaultHandler#endElement(java.lang.String, java.lang.String, java.lang.String)
     */
    @Override
    public void endElement(String uri, String localName, String qName) throws SAXException {
        if (xmlCcm != null && buffer.size() > 0) {
            try {
                xmlCcm.setXmlValue(qName, buffer.toString().trim());
            } catch (Exception e) {
                throw new SAXException(e);
            }
            buffer.reset();
        }
    }

    /*
     * (非 Javadoc)
     * @see org.xml.sax.helpers.DefaultHandler#characters(char[], int, int)
     */
    @Override
    public void characters(char[] ch, int offset, int length) {
        buffer.write(ch, offset, length);
    }

    // CCMを決定するための実装
    private static XmlCcm createCcmCcm(String tagName) throws SAXException {
        // 長さチェック
        if (tagName == null || tagName.length() == 0) {
            return null;
        }

        XmlCcm ccm = null;
        if (DataCcm.TAG_DATA.equals(tagName)) {
            ccm = new DataCcm();
        } else if (RequestCcm.TAG_REQUEST.equals(tagName)) {
            ccm = new RequestCcm();
        } else if (SearchCcm.TAG_SEARCH.equals(tagName)) {
            ccm = new SearchCcm();
        } else if (ServerCcm.TAG_SERVER.equals(tagName)) {
            ccm = new ServerCcm();
        } else if (NodeScanCcm.TAG_NODESCAN.equals(tagName)) {
            ccm = new NodeScanCcm();
        } else if (NodeScanReplyCcm.TAG_NODE.equals(tagName)) {
            ccm = new NodeScanReplyCcm();
        } else if (CcmScanCcm.TAG_CCMSCAN.equals(tagName)) {
            ccm = new CcmScanCcm();
        } else if (CcmScanReplyCcm.TAG_CCMNUM.equals(tagName)) {
            ccm = new CcmScanReplyCcm();
        } else {
            throw new SAXException("can't find CCM type tag.");
        }

        return ccm;
    }

}
