/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.protocol;

import java.io.UnsupportedEncodingException;

import org.w3c.dom.Document;
import org.xml.sax.Attributes;

import com.wabit.uecs.Ccm;
import com.wabit.uecs.UecsConstants;

/**
 * XML形式変換可能なCCM実装クラスです。
 * @author WaBit
 *
 */
public abstract class XmlCcm extends Ccm {

    /** UECSタグ名 */
    public static final String TAG_UECS = "UECS";

    /** ver属性名 */
    public static final String ATTR_VER = "ver";

    /** type属性名 */
    public static final String ATTR_TYPE = "type";

    /** room属性名 */
    public static final String ATTR_ROOM = "room";

    /** region属性名 */
    public static final String ATTR_REGION = "region";

    /** order属性名 */
    public static final String ATTR_ORDER = "order";

    /** priority属性名 */
    public static final String ATTR_PRIORITY = "priority";

    private long receivedTime = Long.MIN_VALUE;

    /**
     * 受信時刻を取得します。
     *
     * @return ミリ秒。未設定の場合は、Long.MIN_VALUEが返ります。
     */
    public long getReceivedTime() {
        return receivedTime;
    }

    /**
     * 受信時刻を設定します。
     *
     * @param receivedTime 受信時刻(msec)
     */
    public void setReceivedTime(long receivedTime) {
        this.receivedTime = receivedTime;
    }

    /**
     * UECS規約にのっとったXML形式メッセージ変換です。
     * @return 変換されたXML文字列
     */
    public String toXml() {
        // Java標準のXML APIでは冗長なXML形式になるため、メッセージサイズ節約目的で
        // UECS専用に実装されたXML変換です。
        StringBuilder sb = new StringBuilder(UecsConstants.MAX_CCM_SIZE);
        sb.append("<?xml version=\"1.0\"?>");
        sb.append("<").append(XmlCcm.TAG_UECS);
        if (getUecsVersion() != null && getUecsVersion().length() > 0) {
            sb.append(" ")
                    .append(ATTR_VER).append("=\"")
                    .append(getUecsVersion()).append("\"");
        }
        sb.append(">");
        appendXml(sb);
        sb.append("</UECS>");
        return sb.toString();
    }

    /*
     * XML用文字列エスケープメソッド
     */
    static final String escape(Object obj) {
        if (obj == null)
            return "";
        String val = obj.toString();
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < val.length(); i++) {
            char c = val.charAt(i);
            switch (c) {
            case '<':
                sb.append("&lt;");
                break;
            case '>':
                sb.append("&gt;");
                break;
            case '&':
                sb.append("&amp;");
                break;
            case '\'':
                sb.append("&apos;");
                break;
            case '\"':
                sb.append("&quot;");
                break;
            default:
                sb.append(c);
            }
        }
        return sb.toString();
    }

    /*
     * (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        return toXml();
    }

    /**
     * XML変換した後のバイトデータを取得します。
     * @return バイト配列
     */
    public byte[] toXmlBytes() {
        try {
            return toXml().getBytes(UecsConstants.DEFAULT_CHARSET);
        } catch (UnsupportedEncodingException e) {
            //ありえないはず
            return null;
        }
    }

    /**
     * CCMの規定ポートを取得します。
     * @return ポート番号
     */
    public abstract int getPort();

    /**
     * XML作成用ビジターメソッドです。
     * @param sb 文字列格納バッファ
     */
    abstract void appendXml(StringBuilder sb);

    /**
     * XML解析してCCM値を吸い出します。
     * @param doc 解析対象DOM
     * @throws Exception 解析に失敗するとスローされます。
     */
    abstract void importXml(Document doc) throws Exception;

    /**
     * SAX方式でXML属性値を解析します。
     * @param tagName タグ名称
     * @param attributes 属性値
     * @throws Exception 解析に失敗するとスローされます。
     */
    abstract void setXmlAttributes(String tagName, Attributes attributes) throws Exception;

    /**
     * SAX方式でXMLタグ値を解析します。
     * @param tagName タグ名称
     * @param value タグ値
     */
    abstract void setXmlValue(String tagName, String value) throws Exception;

}
