/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.device.controller;

import com.wabit.uecs.CcmService;
import com.wabit.uecs.ICcmServiceListener;
import com.wabit.uecs.IUecsNode;
import com.wabit.uecs.UecsConstants;
import com.wabit.uecs.ccm.OprCcm;
import com.wabit.uecs.ccm.RcACcm;
import com.wabit.uecs.ccm.RcMCcm;
import com.wabit.uecs.device.AbstractComponent;
import com.wabit.uecs.device.ComponentConfig;
import com.wabit.uecs.device.IDevice;

/**
 * コントローラ部品を表現するクラスです。
 *
 * @author WaBit
 */
public abstract class ControllerComponent<T extends ComponentConfig> extends AbstractComponent<T> implements
        ICcmServiceListener {

    // リモート受信CCM
    private CcmService oprService;
    // 操作用CCM
    private CcmService rcAService;
    private CcmService rcMService;

    /**
     * コンストラクタ.
     * @param id コンポーネントID
     * @param config 設定値オブジェクト
     */
    public ControllerComponent(String id, T config) {
        super(id, config);
    }

    /**
     * 値を設定します。
     * 同時に連動するCCMの値も変更されます。
     *
     * @param value 値
     */
    @Override
    public void setValue(Number value) {
        if (rcAService != null) {
            rcAService.updateValue(value);
        }
        if (rcMService != null) {
            rcMService.updateValue(value);
        }
        super.setValue(value);
    }

    /**
     * opr.kNN CCMサービスを取得します。(リモート機器の現在値)
     * @return CCMサービス
     */
    public CcmService getOprService() {
        return oprService;
    }

    /**
     * rcA.kNN CCMサービスを取得します。
     * @return CCMサービス
     */
    public CcmService getRcAService() {
        return rcAService;
    }

    /**
     * rcM.kNN CCMサービスを取得します。
     * @return CCMサービス
     */
    public CcmService getRcMService() {
        return rcMService;
    }

    /*
     * (non-Javadoc)
     * @see com.wabit.uecs.device.IComponent#init(com.wabit.uecs.device.IDevice)
     */
    @Override
    public void init(IDevice<?> device) throws Exception {
        setDevice(device);

        // 対応するCCMを登録する
        IUecsNode<?> node = getDevice().getNode();
        ComponentConfig config = getConfig();
        String name = config.getString(ComponentConfig.KEY_COMPONENT_NAME);
        String ccmInfoName = config.getString(ComponentConfig.KEY_CCM_INFO_NAME, "");
        String ccmNodeType = config.getString(ComponentConfig.KEY_CCM_NODE_TYPE, "");
        String unit = config.getString(ComponentConfig.KEY_CCM_UNIT);
        int ccmRoom = config.getInt(ComponentConfig.KEY_CCM_ROOM, 0);
        int ccmRegion = config.getInt(ComponentConfig.KEY_CCM_REGION, 0);
        int ccmOrder = config.getInt(ComponentConfig.KEY_CCM_ORDER, 0);
        int ccmPriority = config.getInt(ComponentConfig.KEY_CCM_PRIORITY, 0);
        int ccmLine = config.getInt(ComponentConfig.KEY_CCM_LINE, 0);

        // リモート機器の制御機器運転状態CCM（opr.kNN）
        OprCcm oprCcm = new OprCcm(ccmInfoName, ccmNodeType, ccmLine);
        oprCcm.setSide(UecsConstants.RECEIVER);
        oprCcm.setUnit(unit);
        oprCcm.setRoom(ccmRoom);
        oprCcm.setRegion(ccmRegion);
        oprCcm.setOrder(ccmOrder);
        oprService = new CcmService(oprCcm);
        oprService.setName(name);
        oprService.addListener(this);
        node.addCcmService(oprService);

        // 遠隔制御指示CCM（rcA.kNN）
        RcACcm rcACcm = new RcACcm(ccmInfoName, ccmNodeType, ccmLine);
        rcACcm.setSide(UecsConstants.SENDER);
        rcACcm.setUnit(unit);
        rcACcm.setRoom(ccmRoom);
        rcACcm.setRegion(ccmRegion);
        rcACcm.setOrder(ccmOrder);
        rcACcm.setPriority(ccmPriority);
        rcAService = new CcmService(rcACcm);
        rcAService.setName(name);
        // 初期状態は不活性
        rcAService.setEnable(false);
        node.addCcmService(rcAService);

        // 遠隔操作指示CCM（rcM.kNN）
        RcMCcm rcMCcm = new RcMCcm(ccmInfoName, ccmNodeType, ccmLine);
        rcMCcm.setSide(UecsConstants.SENDER);
        rcMCcm.setUnit(unit);
        rcMCcm.setRoom(ccmRoom);
        rcMCcm.setRegion(ccmRegion);
        rcMCcm.setOrder(ccmOrder);
        rcMCcm.setPriority(ccmPriority);
        rcMService = new CcmService(rcMCcm);
        rcMService.setName(name);
        // 初期状態は不活性
        rcMService.setEnable(false);
        node.addCcmService(rcMService);

        // 継承クラスの初期化を呼ぶ
        onInit();
    }

}
