/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.device;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.wabit.uecs.IUecsNode;

/**
 * 共通処理を実装した抽象デバイスクラスです。
 *
 * @author WaBit
 * @param <T> デバイス設定オブジェクトクラス
 */
public abstract class AbstractDevice<T extends DeviceConfig> implements IDevice<T> {

    private String id;

    private IUecsNode<?> node;

    private T config;

    private Set<IDeviceListener> listeners = Collections.synchronizedSet(new HashSet<IDeviceListener>());

    private Map<String, IComponent<?>> components = new LinkedHashMap<String, IComponent<?>>();

    /**
     * コンストラクタ.
     * @param config 設定値オブジェクト
     */
    public AbstractDevice(String id, T config) {
        this.id = id;
        this.config = config;
    }

    @Override
    public void addListener(IDeviceListener listener) {
        listeners.add(listener);
    }

    @Override
    public List<IDeviceListener> getListeners() {
        return new ArrayList<IDeviceListener>(listeners);
    }

    @Override
    public boolean removeListener(IDeviceListener listener) {
        return listeners.remove(listener);
    }

    /*
     * (non-Javadoc)
     * @see com.wabit.uecs.device.IDevice#getId()
     */
    @Override
    public String getId() {
        return this.id;
    }

    /*
     * (non-Javadoc)
     * @see com.wabit.uecs.device.IDevice#getNode()
     */
    @Override
    public IUecsNode<?> getNode() {
        return node;
    }

    /*
     * (non-Javadoc)
     * @see com.wabit.uecs.device.IDevice#getConfig()
     */
    @Override
    public T getConfig() {
        return config;
    }

    /*
     * (non-Javadoc)
     * @see com.wabit.uecs.device.IDevice#getComponent(java.lang.String)
     */
    @Override
    public IComponent<?> getComponent(String componentId) {
        return components.get(componentId);
    }

    /*
     * (non-Javadoc)
     * @see com.wabit.uecs.device.IDevice#putComponent(com.wabit.uecs.device.IComponent)
     */
    @Override
    public void addComponent(IComponent<?> component) {
        if (component != null) {
            components.put(component.getId(), component);
        }
    }

    /*
     * (non-Javadoc)
     * @see com.wabit.uecs.device.IDevice#listComponent()
     */
    @Override
    public List<IComponent<?>> listComponents() {
        return new ArrayList<IComponent<?>>(components.values());
    }

    /*
     * (non-Javadoc)
     * @see com.wabit.uecs.device.IDevice#listComponents(java.lang.Class)
     */
    @SuppressWarnings("unchecked")
    public <U extends IComponent<?>> List<U> listComponents(Class<U> cls) {
        List<U> list = new ArrayList<U>();
        for (IComponent<?> compo : listComponents()) {
            if (cls.isInstance(compo)) {
                list.add((U) compo);
            }
        }
        return list;
    }

    /*
     * (non-Javadoc)
     * @see com.wabit.uecs.device.IDevice#removeComponent(java.lang.String)
     */
    @Override
    public IComponent<?> removeComponent(String componentId) {
        return components.remove(componentId);
    }

    @Override
    public void init(IUecsNode<?> node) throws DeviceException {
        this.node = node;

        try {
            onInit();
        } catch (Exception e) {
            notifyException(e);
            throw new DeviceException(e);
        }

        for (IComponent<?> comp : components.values()) {
            try {
                comp.init(this);
            } catch (Exception e) {
                notifyException(e);
                throw new DeviceException(e);
            }
        }

    }

    @Override
    public void start() throws DeviceException {
        try {
            onStart();
        } catch (Exception e) {
            notifyException(e);
            throw new DeviceException(e);
        }
        for(IDeviceListener listener : listeners) {
            listener.deviceStarted(this);
        }
        for (IComponent<?> comp : components.values()) {
            try {
                comp.start();
            } catch (Exception e) {
                notifyException(e);
                throw new DeviceException(e);
            }
        }
    }

    @Override
    public void stop() throws DeviceException {
        for (IComponent<?> comp : components.values()) {
            try {
                comp.stop();
            } catch (Exception e) {
                notifyException(e);
                throw new DeviceException(e);
            }
        }
        try {
            onStop();
        } catch (Exception e) {
            throw new DeviceException(e);
        }
        for(IDeviceListener listener : listeners) {
            listener.deviceStopped(this);
        }
    }

    /**
     * リスナーに開始イベントを伝達します。
     */
    protected void notifyStarted() {
        for(IDeviceListener listener : listeners) {
            listener.deviceStarted(this);
        }
    }

    /**
     * リスナーに停止イベントを伝達します。
     */
    protected void notifyStopped() {
        for(IDeviceListener listener : listeners) {
            listener.deviceStopped(this);
        }
    }


    /**
     * リスナーに例外発生を通知します。
     * @param e 発生した例外
     */
    protected void notifyException(Exception e) {
        for(IDeviceListener listener : listeners) {
            listener.handleDeviceException(this, e);
        }
    }

    /**
     * 初期化(init)実行時に呼び出されます。
     * 主に内部コンポーネントの作成と登録処理を実装します。
     *
     * @throws Exception 処理に失敗するとスローされます。
     */
    abstract protected void onInit() throws Exception;

    /**
     * 起動時の追加処理を実装するメソッドです。
     *
     * @throws Exception 処理に失敗するとスローされます。
     */
    abstract protected void onStart() throws Exception;

    /**
     * 停止時の追加処理を実装するメソッドです。
     *
     * @throws Exception 処理に失敗するとスローされます。
     */
    abstract protected void onStop() throws Exception;
}
