/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs;

/**
 * CCM識別子クラスです。
 *
 * @author WaBit
 *
 */
public final class CcmID {

    /**
     * 大分類記号を定義します。
     *
     * @author WaBit
     *
     */
    public enum MajorDivision {
        /** 制御機器系ノード 'a' を表します。 */
        Actuator("a"),
        /** 操作・コントロール系ノード 'c' を表します 。 */
        Controller("c"),
        /** 計測器系ノード 'm' を表します。 */
        Measure("m"),
        /** 分類不能、設定なしノード 'x' を表します。 */
        Unknown("x");

        private String codeStr;

        private MajorDivision(String code) {
            this.codeStr = code;
        }

        /**
         * コード値を取得します。
         *
         * @return コード値
         */
        public char getCode() {
            return codeStr.charAt(0);
        }

        @Override
        public String toString() {
            return codeStr;
        }

        /**
         * コード値からenumへ変換します。
         *
         * @param code
         * @return enum値
         */
        public static MajorDivision toEnum(char code) {
            switch (code) {
            case 'a':
                return Actuator;
            case 'c':
                return Controller;
            case 'm':
                return Measure;
            default:
                return Unknown;
            }
        }

    }

    private MajorDivision category;
    private String nodeType;
    private String infoName;
    private String code;

    /**
     * コンストラクタ。
     */
    protected CcmID() {}

    /**
     * コンストラクタ
     *
     * @param code 識別子文字列(xxx.kNN)
     */
    public CcmID(String code) {
        if (code == null || code.length() < 3 || code.length() > 19) {
            throw new NodeRuntimeException("ID length violation. code=" + code);
        }
        this.code = code;
        if (code.length() > 4 && code.charAt(code.length() - 4) == '.') {
            int sep = code.length() - 4;
            this.infoName = code.substring(0, sep);
            // .xXX形式のはず
            this.category = MajorDivision.toEnum(code.charAt(sep + 1));
            this.nodeType = code.substring(sep + 2);
        } else {
            this.infoName = code;
            this.category = null;
            this.nodeType = null;
        }
    }

    /**
     * コンストラクタ
     *
     * @param infoName 項目名(xxx)
     * @param nodeType ノード種別(kNN)
     */
    public CcmID(String infoName, String nodeType) {
        if (infoName == null || infoName.length() == 0) {
            throw new NodeRuntimeException("illegal InfoName=" + infoName);
        }
        this.infoName = infoName;
        if (nodeType != null && nodeType.length() > 0) {
            if (nodeType.length() != 3) {
                throw new NodeRuntimeException("illegal NodeType=" + nodeType);
            } else {
                // .xXX形式のはず
                this.category = MajorDivision.toEnum(nodeType.charAt(0));
                this.nodeType = nodeType;
                this.code = this.infoName + "." + nodeType;
            }
        } else if (nodeType.length() > 0) {
            this.code = this.infoName + "." + nodeType;
        } else {
            this.code = this.infoName;
        }
    }

    /**
     * 項目名を取得します。
     *
     * @return 項目名
     */
    public String getInfoName() {
        return infoName;
    }

    /**
     * 大分類を取得します。
     *
     * @return 大分類
     */
    public MajorDivision getMajorDivision() {
        return category;
    }

    /**
     * ノード種類（機器種類）を取得します。
     *
     * @return ノード種類文字列(2文字)
     */
    public String getNodeType() {
        return nodeType;
    }

    /**
     * コード値を取得します。
     *
     * @return 識別子文字列(xxx.kNN)
     */
    public String getCode() {
        return code;
    }

    /*
     * (non-Javadoc)
     *
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        return code;
    }

    /*
     * (non-Javadoc)
     *
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
        return code.hashCode();
    }

    /*
     * (non-Javadoc)
     *
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(Object o) {
        if (super.equals(o)) {
            return true;
        }
        if (o instanceof CcmID) {
            return this.code.equals(((CcmID) o).code);
        } else {
            return false;
        }
    }
}
