/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs;

/**
 * ノードの動作モードを定義するEnumです。
 * @author WaBit
 */
public enum ActionMode {

    /** 自律動作モード */
    Autonomy(UecsConstants.MODE_AUTONOMY),
    /** rcAモード */
    rcA(UecsConstants.MODE_rcA),
    /** WEB強制操作モード */
    WEB(UecsConstants.MODE_WEB),
    /** rcMモード */
    rcM(UecsConstants.MODE_rcM),
    /** インターロックモード */
    Interlock(UecsConstants.MODE_INTERLOCK),
    /** 強制停止モード */
    Outage(UecsConstants.MODE_OUTAGE),
    /** スタンドアローンモード */
    Standalone(UecsConstants.MODE_STANDALONE)
    ;

    private int modeBits;

    private ActionMode(int val) {
        this.modeBits = val;
    }

    /**
     * 動作モードの4bit表現値を取得します。
     * @return ビット値（機器動作状態CCMの16bit-19bit範囲の値）
     */
    public int getBits() {
        return modeBits;
    }

    /**
     * ビット値からEnumを取得します。
     * @param modeBits ビット値（機器動作状態CCMの16bit-19bit範囲の値）
     * @return モードEnum値
     */
    public static ActionMode getMode(int modeBits) {
        switch (modeBits) {
        case UecsConstants.MODE_AUTONOMY :
            return Autonomy;
        case UecsConstants.MODE_STANDALONE :
            return Standalone;
        case UecsConstants.MODE_rcA :
            return rcA;
        case UecsConstants.MODE_rcM :
            return rcM;
        case UecsConstants.MODE_INTERLOCK :
            return Interlock;
        case UecsConstants.MODE_WEB :
            return WEB;
        case UecsConstants.MODE_OUTAGE :
            return Outage;
        default :
            return null;
        }
    }


    /**
     * 優先度の判定を行います。
     * @param mode モードEnum値
     * @return 自身の優先度が高い場合、もしくは同一の場合はtrue. 低ければfalse.
     */
    public boolean isPrior(ActionMode mode) {
        // 一番低いスタンドアローンが一番高いビット値なので、特別扱い
        if (Standalone == mode) {
            return true;
        }
        if (this == Standalone) {
            return false;
        }
        // その他は値の大小で比較できる
        return getBits() >= mode.getBits();
    }
}
