/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.device.sensor;

import com.wabit.uecs.Ccm;
import com.wabit.uecs.CcmID;
import com.wabit.uecs.CcmLevel;
import com.wabit.uecs.CcmService;
import com.wabit.uecs.ICcmServiceListener;
import com.wabit.uecs.IUecsNode;
import com.wabit.uecs.UecsConstants;
import com.wabit.uecs.device.AbstractComponent;
import com.wabit.uecs.device.ComponentConfig;
import com.wabit.uecs.device.IDevice;

/**
 * センサー部品を表現するクラスです。
 * @author WaBit
 */
public abstract class SensorComponent<T extends ComponentConfig> extends AbstractComponent<T> {

	// 連動するセンサー値送信CCM
	private CcmService ccmService;

	/**
	 * コンストラクタ.
	 * @param id コンポーネントID
	 * @param config 設定値オブジェクト
	 */
    public SensorComponent(String id, T config) {
        super(id, config);
    }

    /**
     * デバイス値を設定します。
     * 同時に関連するCCMの値も更新されます。
     *
     * @param value 値
     */
    @Override
    public void setValue(Number value) {
        getCcmService().updateValue(value);
    	super.setValue(value);
    }

    /**
     * センサーに対応するCCMサービスを取得します。
     * @return CCMサービス
     */
    public CcmService getCcmService() {
        return ccmService;
    }

    /*
     * (non-Javadoc)
     * @see com.wabit.uecs.device.IComponent#init(com.wabit.uecs.device.IDevice)
     */
    @Override
    public void init(IDevice<?> device) throws Exception {
    	setDevice(device);

        IUecsNode<?> node = getDevice().getNode();

        ComponentConfig config = getConfig();
        String name = config.getString(ComponentConfig.KEY_COMPONENT_NAME);
        String ccmInfoName = config.getString(ComponentConfig.KEY_CCM_INFO_NAME, "");
        String ccmNodeType = config.getString(ComponentConfig.KEY_CCM_NODE_TYPE, "");
        String ccmLevel = config.getString(ComponentConfig.KEY_CCM_LEVEL);
        String unit = config.getString(ComponentConfig.KEY_CCM_UNIT);
        char side = config.getString(ComponentConfig.KEY_CCM_SIDE, ""+UecsConstants.RECEIVER).charAt(0);
        int ccmRoom = config.getInt(ComponentConfig.KEY_CCM_ROOM, 0);
        int ccmRegion = config.getInt(ComponentConfig.KEY_CCM_REGION, 0);
        int ccmOrder = config.getInt(ComponentConfig.KEY_CCM_ORDER, 0);
        int ccmPriority = config.getInt(ComponentConfig.KEY_CCM_PRIORITY, 0);
        int ccmCast = config.getInt(ComponentConfig.KEY_CCM_CAST, 0);
        String ccmId = ccmInfoName;
        if (ccmNodeType != null && ccmNodeType.length() > 0) {
        	ccmId += "." + ccmNodeType;
        }
        Ccm ccm = new Ccm();
        ccm.setSide(side);
        ccm.setType(new CcmID(ccmId));
        ccm.setUnit(unit);
        ccm.setCast(ccmCast);
        ccm.setRoom(ccmRoom);
        ccm.setRegion(ccmRegion);
        ccm.setOrder(ccmOrder);
        ccm.setPriority(ccmPriority);
        // Bレベルの場合、強制的にpriorityが30になる
        ccm.setLevel(new CcmLevel(ccmLevel));

        ccmService = new CcmService(ccm);
        ccmService.setName(name);
        if (side == UecsConstants.RECEIVER) {
            ccmService.addListener(new InnerListener());
        }

        node.addCcmService(ccmService);

        onInit();
    }

    /*
     * (non-Javadoc)
     * @see com.wabit.uecs.device.IComponent#start()
     */
    @Override
    public void start() throws Exception {
    	// 初期値設定
        getCcmService().updateValue(getValue());
    	onStart();
    	notifyStarted();
    }

    /*
     * センサーCCM受信動作用リスナー
     *
     */
    private class InnerListener implements ICcmServiceListener {

        @Override
        public void ccmValueChanged(CcmService source, Ccm value) {
            SensorComponent.super.setValue(value.getNumberValue());
        }

        @Override
        public void ccmExpired(CcmService source, Ccm value) {
            SensorComponent.super.setValue(null);
        }

        @Override
        public void ccmSent(CcmService source, Ccm value) {
            // TODO 何もしない
        }

        @Override
        public void ccmReceived(CcmService source, Ccm value) {
            // TODO 何もしない
        }

    }

}
