/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.device.actuator;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

/**
 * 一日を時間帯分割して動作させるためのアクション
 * 複数のアクションを内包します。
 * @author WaBit
 *
 */
public class DailySchedulerAction extends AbstractCyclicAction {

    private class Schedule {
        int startHM;
        int endHM;
        IActuatorAction action;
    }

    private List<Schedule> schedules = new ArrayList<Schedule>();
    private IActuatorAction resetAction;

    /**
     * デフォルトコンストラクタ(10秒隔動作)
     */
    public DailySchedulerAction() {
        super(10L * 1000L);
    }

    /**
     * 動作チェック間隔を指定したコンストラクタ
     * @param checkInterval 動作間隔(msec)
     */
    public DailySchedulerAction(long checkInterval) {
        super(checkInterval);
    }

    /**
     * スケジュール実行されない時間帯のリセット動作を設定します。
     * @param action アクション
     */
    public void setResetAction(IActuatorAction action) {
        this.resetAction = action;
    }

    /*
     * (non-Javadoc)
     * @see com.wabit.uecs.device.actuator.IActuatorAction#onStart(com.wabit.uecs.device.actuator.ActuatorComponent)
     */
    @Override
    public void onStart(ActuatorComponent<?> component) throws Exception {
        for (Schedule sc : schedules) {
            sc.action.onStart(component);
        }
        if (resetAction != null) {
            resetAction.onStart(component);
        }
        super.onStart(component);
    }

    /*
     * (non-Javadoc)
     * @see com.wabit.uecs.device.actuator.IActuatorAction#onStop(com.wabit.uecs.device.actuator.ActuatorComponent)
     */
    @Override
    public void onStop(ActuatorComponent<?> component) throws Exception {
        super.onStop(component);
        for (Schedule sc : schedules) {
            sc.action.onStop(component);
        }
        if (resetAction != null) {
            resetAction.onStop(component);
        }
    }

    /*
     * (non-Javadoc)
     * @see com.wabit.uecs.device.actuator.IActuatorAction#execute(com.wabit.uecs.device.actuator.ActuatorComponent)
     */
    @Override
    public void execute(ActuatorComponent<?> component) throws Exception {
        for (Schedule sc : schedules) {
            Calendar cal = Calendar.getInstance();
            int hm = cal.get(Calendar.HOUR_OF_DAY) * 10000
                    + cal.get(Calendar.MINUTE) * 100
                    + cal.get(Calendar.SECOND);
            if (sc.startHM < sc.endHM) {
                if (hm >= sc.startHM && hm <= sc.endHM) {
                    sc.action.execute(component);
                } else if (resetAction != null) {
                    resetAction.execute(component);
                }
            } else if (sc.startHM > sc.endHM) {
                // 日またぎ時間帯
                if ((hm >= sc.startHM && hm <= 235959)
                    || (hm >= 000000 && hm <= sc.endHM)){
                    sc.action.execute(component);
                } else if (resetAction != null) {
                    resetAction.execute(component);
                }
            }
        }
    }

    /**
     * スケジュールを追加します。
     * @param action 実行アクション
     * @param startHour 動作開始時
     * @param startMin 動作開始分
     * @param endHour 動作終了時
     * @param endMin 動作終了分
     */
    public void addSchedule(IActuatorAction action, int startHour, int startMin, int endHour, int endMin) {
        Schedule sc = new Schedule();
        sc.action = action;
        sc.startHM = startHour * 10000 + startMin * 100;
        sc.endHM = endHour * 10000 + endMin * 100;
        schedules.add(sc);
    }


}
