/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.pi.webui.nodeconfig;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.TimeZone;
import java.util.concurrent.Callable;

import javax.servlet.http.HttpServletResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.wicket.authroles.authorization.strategies.role.Roles;
import org.apache.wicket.authroles.authorization.strategies.role.annotations.AuthorizeInstantiation;
import org.apache.wicket.extensions.markup.html.form.DateTextField;
import org.apache.wicket.markup.html.WebMarkupContainer;
import org.apache.wicket.markup.html.basic.Label;
import org.apache.wicket.markup.html.form.Button;
import org.apache.wicket.markup.html.form.CheckBox;
import org.apache.wicket.markup.html.form.Form;
import org.apache.wicket.markup.html.form.PasswordTextField;
import org.apache.wicket.markup.html.form.RadioChoice;
import org.apache.wicket.markup.html.form.RequiredTextField;
import org.apache.wicket.markup.html.form.TextField;
import org.apache.wicket.markup.html.form.upload.FileUploadField;
import org.apache.wicket.model.CompoundPropertyModel;
import org.apache.wicket.model.IModel;
import org.apache.wicket.validation.validator.PatternValidator;
import org.apache.wicket.validation.validator.RangeValidator;
import org.apache.wicket.validation.validator.StringValidator;

import com.wabit.uecs.NodeConfig;
import com.wabit.uecs.pi.AppConstants;
import com.wabit.uecs.pi.UecsPiNode;
import com.wabit.uecs.pi.UecsPiNodeConfig;
import com.wabit.uecs.pi.db.DatabaseManager;
import com.wabit.uecs.pi.db.DatabaseUtils;
import com.wabit.uecs.pi.util.BytesUtils;
import com.wabit.uecs.pi.util.DateTimeUtils;
import com.wabit.uecs.pi.util.MessageCode;
import com.wabit.uecs.pi.util.MessageUtils;
import com.wabit.uecs.pi.webui.LayoutPage;
import com.wabit.uecs.pi.webui.LocaleChoice;
import com.wabit.uecs.pi.webui.TimeZoneChoice;
import com.wabit.uecs.pi.webui.WebUIApplication;

/**
 * WebUIのノード設定画面機能を実装したクラスです。
 *
 * @author WaBit
 */
@AuthorizeInstantiation({ Roles.ADMIN })
public class NodeConfigPage extends LayoutPage {

    private static final long serialVersionUID = 9036704791217192197L;

    private static final String IP_PATTERN = "^\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}$";

    private Log logger = LogFactory.getLog(getClass());
    private NodeSetting nodeSettings;

    private final WebMarkupContainer editIpForm = new WebMarkupContainer(
            "editIpForm");
    private final WebMarkupContainer staticIpForm = new WebMarkupContainer(
            "staticIpForm");

    @Override
    protected void onInitialize() {
        super.onInitialize();
        nodeSettings = createModel();

        UecsPiNode node = WebUIApplication.getNodeInstance();
        UecsPiNodeConfig conf = (UecsPiNodeConfig) node.getConfig();
        nodeSettings.setNodeName(node.getName());
        nodeSettings.setNodeType(conf.getString(UecsPiNodeConfig.KEY_NODE_TYPE,
                ""));
        nodeSettings.setIpAddress(node.getIpAddress().getHostAddress());
        nodeSettings.setPassword(conf.getString(
                UecsPiNodeConfig.KEY_ADMIN_PASSWORD, ""));
        nodeSettings.setCurrentDate(new Date());
        nodeSettings.setIsChangeCurrentDate(false);
        nodeSettings.setIpType(conf.getString(
                UecsPiNodeConfig.KEY_NODE_IP_TYPE, ""));
        nodeSettings.setIpMask(conf.getString(
                UecsPiNodeConfig.KEY_NODE_IP_MASK, ""));
        nodeSettings.setGateway(conf.getString(
                UecsPiNodeConfig.KEY_NODE_IP_GATEWAY, ""));
        nodeSettings.setDnsServer(conf.getString(
                UecsPiNodeConfig.KEY_NODE_DNS_SERVER, ""));
        nodeSettings.setRoom(conf.getInt(UecsPiNodeConfig.KEY_NODE_ROOM, 0));
        nodeSettings
                .setRegion(conf.getInt(UecsPiNodeConfig.KEY_NODE_REGION, 0));
        nodeSettings.setOrder(conf.getInt(UecsPiNodeConfig.KEY_NODE_ORDER, 0));
        nodeSettings.setPriority(conf.getInt(
                UecsPiNodeConfig.KEY_NODE_PRIORITY, 0));
        nodeSettings.setIsRcvDateTime(conf.getBoolean(
                UecsPiNodeConfig.KEY_RECEIVE_DATETIME, false));
        nodeSettings.setIsSndDateTime(conf.getBoolean(
                UecsPiNodeConfig.KEY_SEND_DATETIME, false));
        nodeSettings.setLocale(conf.getString(
                UecsPiNodeConfig.KEY_LOCALE, Locale.getDefault().toLanguageTag()));
        nodeSettings.setTimeZone(conf.getString(
                UecsPiNodeConfig.KEY_TIME_ZONE, TimeZone.getDefault().getID()));

        Form<NodeSetting> form = new Form<NodeSetting>("form");

        IModel<NodeSetting> model = new CompoundPropertyModel<NodeSetting>(
                nodeSettings);
        form.setDefaultModel(model);
        form.add(new RequiredTextField<String>("nodeName")
                .add(new StringValidator(1, 50)));
        form.add(new RequiredTextField<String>("nodeType")
                .add(new StringValidator(1, 3)));
        form.add(new RequiredTextField<Integer>("room", Integer.class)
                .add(new RangeValidator<Integer>(0, 127)));
        form.add(new RequiredTextField<Integer>("region", Integer.class)
                .add(new RangeValidator<Integer>(0, 127)));
        form.add(new RequiredTextField<Integer>("order", Integer.class)
                .add(new RangeValidator<Integer>(0, 3000)));
        form.add(new RequiredTextField<Integer>("priority", Integer.class)
                .add(new RangeValidator<Integer>(0, 29)));
        form.add(new DateTextField("currentDate", DateTimeUtils.DATETIME_FORMAT));
        form.add(new PasswordTextField("password").setRequired(false).add(
                new StringValidator(1, 50)));
        form.add(new Label("macAddress", BytesUtils.toHexString(node.getMacAddress(), "-")));
        form.add(new Label("firmwareVersion", conf.getString(
                UecsPiNodeConfig.KEY_FIRMWARE_VERSION, "")));
        form.add(new CheckBox("isChangeCurrentDate"));
        form.add(new CheckBox("isRcvDateTime"));
        form.add(new CheckBox("isSndDateTime"));

        staticIpForm.add(new RequiredTextField<String>("ipAddress")
                .add(new PatternValidator(IP_PATTERN)));
        staticIpForm.add(new RequiredTextField<String>("ipMask")
                .add(new PatternValidator(IP_PATTERN)));
        staticIpForm.add(new TextField<String>("gateway")
                .add(new PatternValidator(IP_PATTERN)));
        staticIpForm.add(new TextField<String>("dnsServer")
                .add(new PatternValidator(IP_PATTERN)));
        if ("static".equals(conf.getString(UecsPiNodeConfig.KEY_NODE_IP_TYPE,
                ""))) {
            staticIpForm.setVisible(true);
        } else {
            staticIpForm.setVisible(false);
        }
        editIpForm.add(staticIpForm);

        List<String> ipTypes = Arrays.asList("dhcp", "static");
        editIpForm.add(new RadioChoice<String>("ipType", ipTypes) {
            private static final long serialVersionUID = 1L;

            @Override
            protected boolean localizeDisplayValues() {
                return true;
            }

            @Override
            protected boolean wantOnSelectionChangedNotifications() {
                return true;
            }

            @Override
            protected void onSelectionChanged(final Object val) {
                if ("dhcp".equals(val)) {
                    staticIpForm.setVisible(false);
                } else {
                    staticIpForm.setVisible(true);
                }
            }
        });
        form.add(editIpForm);

        // モバイルルータモデルとノーマルモデルの表示切替
        String networkType = WebUIApplication.get().getInitParameter("ipReadonly");
        if (networkType != null && networkType.equalsIgnoreCase("true")) {
            editIpForm.setVisible(false);
        } else {
            editIpForm.setVisible(true);
        }

        // ロケールリスト
        form.add(new LocaleChoice("locale"));

        // タイムゾーン
        form.add(new TimeZoneChoice("timeZone"));

        form.add(new Button("saveButton") {
            private static final long serialVersionUID = 1L;

            @Override
            public void onSubmit() {
                try {
                    onConfigChanged();
                } catch (Throwable e) {
                    error(MessageUtils.getMessage(MessageCode.ERROR));
                    logger.error("restart error.", e);
                }
            }
        });

        form.add(new Button("shutdownButton") {
            private static final long serialVersionUID = 1L;

            @Override
            public void onSubmit() {
                try {
                    shutdownOS();
                    success(MessageUtils.getMessage(MessageCode.OS_SHUTDOWN));
                } catch (Throwable e) {
                    error(MessageUtils.getMessage(MessageCode.ERROR));
                    logger.error("OS shutdown error.", e);
                }
            }
        });

        form.add(new Button("rebootButton") {
            private static final long serialVersionUID = 1L;

            @Override
            public void onSubmit() {
                try {
                    rebootOS();
                    success(MessageUtils.getMessage(MessageCode.OS_REBOOT));
                } catch (Throwable e) {
                    error(MessageUtils.getMessage(MessageCode.ERROR));
                    logger.error("OS reboot error.", e);
                }
            }
        });

        add(form);

        final FileUploadField uploadField = new FileUploadField("uploadFile");
        Form<Void> uploadForm = new Form<Void>("configFileForm");
        uploadForm.add(uploadField);
        uploadForm.add(new Button("uploadButton") {
            private static final long serialVersionUID = 1L;

            @Override
            public void onSubmit() {
                try {
                    if (uploadField.getFileUpload() != null) {
                        final UecsPiNode node = WebUIApplication.getNodeInstance();

                        DatabaseManager.callInTransaction(
                                new Callable<Void>() {
                                    public Void call() throws Exception {
                                        InputStream input = uploadField.getFileUpload().getInputStream();

                                        DatabaseUtils.readConfigXml(input);

                                        // 上書き不可項目を再セット
                                        UecsPiNodeConfig conf = node.getConfig();
                                        for (String key : conf.getFixKeys()) {
                                            DatabaseUtils.saveNodeConfig(key, conf.getString(key));
                                        }
                                        return null;
                                    }
                                });

                        // ノードを再起動する
                        node.restart();
                        success(MessageUtils.getMessage(MessageCode.SAVED));
                    }
                } catch (Exception e) {
                    error(MessageUtils.getMessage(MessageCode.ERROR));
                    logger.error("upload error.", e);
                }
            }
        });

        uploadForm.add(new Button("downloadButton") {
            private static final long serialVersionUID = 1L;

            @Override
            public void onSubmit() {
                OutputStream out = null;
                try {
                    HttpServletResponse httpRes = (HttpServletResponse) getResponse().getContainerResponse();
                    httpRes.setContentType("text/xml");
                    httpRes.setHeader("Content-Disposition", "attachment; filename=node-config.xml");
                    out = httpRes.getOutputStream();
                    DatabaseUtils.writeConfigXml(out);
                    out.flush();
                    out.close();
                } catch (Exception e) {
                    error(MessageUtils.getMessage(MessageCode.ERROR));
                    logger.error("download error.", e);
                }
            }
        });
        add(uploadForm);

        // 子クラス実装処理
        onInit(form);
    }

    // hook
    protected void onConfigChanged() throws Exception {
        DatabaseManager.callInTransaction(
                new Callable<Void>() {
                    public Void call() throws Exception {
                        NodeConfig saveConf = new NodeConfig();

                        // 既登録値をコピーしておく
                        saveConf.putAll(WebUIApplication.getNodeInstance().getConfig());

                        saveConf.setString(UecsPiNodeConfig.KEY_NODE_NAME, nodeSettings.getNodeName().trim());
                        saveConf.setString(UecsPiNodeConfig.KEY_NODE_TYPE, nodeSettings.getNodeType().trim());
                        saveConf.setString(UecsPiNodeConfig.KEY_NODE_IP_TYPE, nodeSettings.getIpType());
                        if (nodeSettings.getPassword() != null && nodeSettings.getPassword().length() > 0) {
                            saveConf.setString(UecsPiNodeConfig.KEY_ADMIN_PASSWORD, nodeSettings.getPassword());
                        }
                        saveConf.setInt(UecsPiNodeConfig.KEY_NODE_ROOM, nodeSettings.getRoom());
                        saveConf.setInt(UecsPiNodeConfig.KEY_NODE_REGION, nodeSettings.getRegion());
                        saveConf.setInt(UecsPiNodeConfig.KEY_NODE_ORDER, nodeSettings.getOrder());
                        saveConf.setInt(UecsPiNodeConfig.KEY_NODE_PRIORITY, nodeSettings.getPriority());
                        saveConf.setBoolean(UecsPiNodeConfig.KEY_RECEIVE_DATETIME, nodeSettings.getIsRcvDateTime());
                        saveConf.setBoolean(UecsPiNodeConfig.KEY_SEND_DATETIME, nodeSettings.getIsSndDateTime());
                        saveConf.setString(UecsPiNodeConfig.KEY_LOCALE, nodeSettings.getLocale());
                        saveConf.setString(UecsPiNodeConfig.KEY_TIME_ZONE, nodeSettings.getTimeZone());

                        if (nodeSettings.getIsChangeCurrentDate()
                                && nodeSettings.getCurrentDate() != null) {
                            changeCurrentDate(nodeSettings.getCurrentDate());
                        }

                        UecsPiNode node = WebUIApplication.getNodeInstance();
                        String oldIpAddress = node.getIpAddress().toString().replace("/", "");
                        String oldSubnetMask = ((UecsPiNodeConfig) node.getConfig())
                                .getString(UecsPiNodeConfig.KEY_NODE_IP_MASK, "");
                        String oldGateway = ((UecsPiNodeConfig) node.getConfig())
                                .getString(UecsPiNodeConfig.KEY_NODE_IP_GATEWAY, "");
                        String oldIpType = ((UecsPiNodeConfig) node.getConfig())
                                .getString(UecsPiNodeConfig.KEY_NODE_IP_TYPE, "");
                        String oldDnsServer = ((UecsPiNodeConfig) node.getConfig())
                                .getString(UecsPiNodeConfig.KEY_NODE_DNS_SERVER, "");

                        // サーバIPの設定
                        String newIpType = nodeSettings.getIpType();
                        String newIpAddress = nodeSettings.getIpAddress();
                        String newSubnetMask = nodeSettings.getIpMask();
                        String newGateway = nodeSettings.getGateway();
                        String newDnsServer = nodeSettings.getDnsServer();

                        // 子クラス実装
                        onSave(nodeSettings, saveConf);

                        if (newIpType.equals(AppConstants.IP_TYPE_DHCP)) {
                            // dhcpを使用する
                            if (oldIpType.equals(AppConstants.IP_TYPE_STATIC)) {
                                DatabaseUtils.saveNodeConfig(saveConf);
                                // ノードを停止して再起動に備える
                                WebUIApplication.getNodeInstance().stop();
                                // DHCPへ変更スクリプト呼び出し
                                changeIpAddressToDhcp();
                                // 遅延再起動
                                WebUIApplication.getNodeInstance().restartAsync(10000L);
                                success(MessageUtils.getMessage(MessageCode.IP_CHANGED));
                                return null;
                            }
                        } else if (oldIpType.equals(AppConstants.IP_TYPE_DHCP)
                                || !oldIpAddress.equals(newIpAddress)
                                || !oldSubnetMask.equals(newSubnetMask)
                                || !oldGateway.equals(newGateway)
                                || !oldDnsServer.equals(newDnsServer)) {

                            // staticを使用する
                            if (newDnsServer == null) {
                                newDnsServer = "";
                            }
                            replaceDnsServerDefinition(newDnsServer);
                            if (newGateway == null) {
                                newGateway = "";
                            }

                            // 自ノードIP以外はJavaAPIから取れないので保存しておく
                            saveConf.setString(UecsPiNodeConfig.KEY_NODE_IP_MASK, newSubnetMask);
                            saveConf.setString(UecsPiNodeConfig.KEY_NODE_IP_GATEWAY, newGateway);
                            saveConf.setString(UecsPiNodeConfig.KEY_NODE_DNS_SERVER, newDnsServer);

                            DatabaseUtils.saveNodeConfig(saveConf);

                            // ノードを停止して再起動に備える
                            WebUIApplication.getNodeInstance().stop();
                            // IPアドレスを変更
                            changeIpAddress(newIpAddress, newSubnetMask, newGateway);
                            // 遅延再起動
                            WebUIApplication.getNodeInstance().restartAsync(5000L);
                            success(MessageUtils.getMessage(MessageCode.IP_CHANGED));
                            return null;
                        }
                        DatabaseUtils.saveNodeConfig(saveConf);

                        // 表示言語切り替え
                        WebUIApplication webApp = (WebUIApplication) getApplication();
                        if (!webApp.getDefaultLocale().toLanguageTag()
                                .equals(nodeSettings.getLocale())) {
                            webApp.setDefaultLocale(Locale.forLanguageTag(nodeSettings.getLocale()));
                            getSession().setLocale(webApp.getDefaultLocale());
                        }

                        // ノードを再起動する
                        WebUIApplication.getNodeInstance().restart();

                        success(MessageUtils.getMessage(MessageCode.SAVED));
                        return null;
                    }
                });
    }

    /**
     * モデルオブジェクト生成
     * @return
     */
    protected NodeSetting createModel() {
        return new NodeSetting();
    }

    /**
     * 継承クラスで追加の設定項目処理を行います。
     * @param model
     */
    protected void onInit(Form<NodeSetting> form) {

    }

    /**
     * 継承クラスで追加のデータ項目保存処理を実装します。
     * @param model
     */
    protected void onSave(NodeSetting source, NodeConfig saveConf) {

    }

    /**
     * シェルスクリプトを呼び出し、IPアドレス、ゲートウェイを変更する
     *
     * @param newIpAddress
     * @param newGateway
     * @throws IOException
     */
    private void changeIpAddress(String newIpAddress, String newSubnetMask,
            String newGateway) throws IOException {
        String cmd = "sh " + AppConstants.SCRIPT_CHANGE_IP_STATIC + " "
                + newIpAddress + " " + newSubnetMask + " " + newGateway;
        Runtime rt = Runtime.getRuntime();
        rt.exec(cmd);
    }

    /**
     * シェルスクリプトを呼び出し、DHCPによるIPアドレス取得に変更する。
     *
     * @throws IOException
     */
    private void changeIpAddressToDhcp() throws IOException {
        String cmd = null;
        cmd = "sh " + AppConstants.SCRIPT_CHANGE_IP_DHCP;
        Runtime rt = Runtime.getRuntime();
        rt.exec(cmd);
    }

    /**
     * シェルスクリプトを呼び出し、DNSサーバの設定ファイルを編集する。
     *
     * @param newDnsServer
     * @throws IOException
     */
    private void replaceDnsServerDefinition(String newDnsServer)
            throws IOException {
        String cmd = "sh " + AppConstants.SCRIPT_SET_DNS + " " + newDnsServer;
        Runtime rt = Runtime.getRuntime();
        rt.exec(cmd);
    }

    /**
     * サーバの時刻を入力された時間に設定する
     *
     * @param currentDate
     * @throws Exception
     */
    private void changeCurrentDate(Date date) throws Exception {

        Runtime rt = Runtime.getRuntime();
        // 指定されたフォーマットのオブジェクト生成
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        rt.exec(new String[] { "sh", AppConstants.SCRIPT_CHANGE_DATETIME,
                sdf.format(date) });
    }

    /**
     * OSシャットダウンを行います。
     * @throws Exception
     */
    private void shutdownOS() throws Exception {
        WebUIApplication.getNodeInstance().stop();
        String cmd = AppConstants.SCRIPT_OS_SHUTDOWN;
        Runtime rt = Runtime.getRuntime();
        rt.exec(cmd);
    }

    /**
     * OS再起動を行います。
     * @throws Exception
     */
    private void rebootOS() throws Exception {
        WebUIApplication.getNodeInstance().stop();
        String cmd = AppConstants.SCRIPT_OS_REBOOT;
        Runtime rt = Runtime.getRuntime();
        rt.exec(cmd);
    }
}
