/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.pi.webui.log;

import java.util.ArrayList;
import java.util.List;

import org.apache.wicket.authroles.authorization.strategies.role.Roles;
import org.apache.wicket.authroles.authorization.strategies.role.annotations.AuthorizeInstantiation;
import org.apache.wicket.datetime.markup.html.basic.DateLabel;
import org.apache.wicket.markup.html.basic.Label;
import org.apache.wicket.markup.html.form.Button;
import org.apache.wicket.markup.html.form.DropDownChoice;
import org.apache.wicket.markup.html.form.StatelessForm;
import org.apache.wicket.markup.html.list.ListItem;
import org.apache.wicket.markup.html.list.ListView;
import org.apache.wicket.model.Model;
import org.apache.wicket.model.ResourceModel;

import com.wabit.uecs.pi.UecsPiLogger;
import com.wabit.uecs.pi.db.LogEntity;
import com.wabit.uecs.pi.util.MessageCode;
import com.wabit.uecs.pi.util.MessageUtils;
import com.wabit.uecs.pi.webui.LayoutPage;

/**
 * WebUIの状態ログ画面機能を実装したクラスです。
 *
 * @author WaBit
 */
@AuthorizeInstantiation({Roles.ADMIN})
public class LogPage extends LayoutPage {

    private static final long serialVersionUID = 3408233779384475934L;

    private static final String CATEGORY_ALL = "ALL";

    private static final int LIST_SIZE = 100;
    private ListView<LogEntity> logListView;
    private String category;

    @Override
    protected void onInitialize() {
        super.onInitialize();

        List<String> choiceList = getCategoryList();
        if (choiceList != null && choiceList.size() > 0) {
            category = choiceList.get(0);
        }

        DropDownChoice<String> categoryChoice = new DropDownChoice<String>("selectedCategory", Model.of(category), choiceList) {
            private static final long serialVersionUID = 1L;
            protected boolean localizeDisplayValues() {
                return true;
            }
            @Override
            protected boolean wantOnSelectionChangedNotifications() {
                return true;
            }
            @Override
            protected void onSelectionChanged(String selected) {
                category = selected;
                logListView.modelChanging();
                logListView.setModelObject(fetchLogs(category, LIST_SIZE));
                logListView.modelChanged();
            }
        };
        add(categoryChoice);

        StatelessForm<LogPage> form = new StatelessForm<LogPage>("form");
        form.add(new Button("updateButton") {
            @Override
            public void onSubmit() {
                logListView.modelChanging();
                logListView.setModelObject(fetchLogs(category, LIST_SIZE));
                logListView.modelChanged();
            }
        });
        form.add(new Button("clearButton") {
            @Override
            public void onSubmit() {
                logListView.modelChanging();
                UecsPiLogger.clearLogs();
                logListView.setModelObject(new ArrayList<LogEntity>(0));
                logListView.modelChanged();
                success(MessageUtils.getMessage(MessageCode.CLEARED));
            }
        });
        add(form);

        List<LogEntity> logs = fetchLogs(category, LIST_SIZE);
        logListView = new ListView<LogEntity>("logList", logs) {
            private static final long serialVersionUID = 1L;

            @Override
            protected void populateItem(ListItem<LogEntity> item) {
                LogEntity entity = item.getModelObject();
                //item.add(new Label("logNo", entity.getLogNo()));
                item.add(DateLabel.forDatePattern("time", Model.of(entity.getTime()), "yyyy-MM-dd HH:mm:ss z"));
                item.add(new Label("category", new ResourceModel(entity.getCategory())));
                item.add(new Label("message", entity.getMessage()));

            }
        };
        add(logListView);
    }

    /*
     * カテゴリ一覧を取得します。
     */
    private List<String> getCategoryList() {
        ArrayList<String> list = new ArrayList<String>();
        list.add(UecsPiLogger.CATEGORY_NODE);
        list.add(UecsPiLogger.CATEGORY_DEVICE);
        list.add(UecsPiLogger.CATEGORY_OTHER);
        list.add(CATEGORY_ALL); // 全て
        return list;
    }

    /*
     * ログ取得を行います。
     */
    private List<LogEntity> fetchLogs(String category, int size) {
        if (CATEGORY_ALL.equals(category)) {
            return UecsPiLogger.getLogs(size);
        } else {
            return UecsPiLogger.getLogs(category, size);
        }
    }
}
