/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.pi.device.gpio;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.pi4j.io.gpio.GpioPinDigitalInput;
import com.pi4j.io.gpio.GpioPinInput;
import com.pi4j.io.gpio.Pin;
import com.pi4j.io.gpio.PinPullResistance;
import com.pi4j.io.gpio.event.GpioPinDigitalStateChangeEvent;
import com.pi4j.io.gpio.event.GpioPinListenerDigital;
import com.pi4j.io.gpio.event.PinEventType;
import com.wabit.uecs.ActionMode;
import com.wabit.uecs.pi.device.PiActuatorBase;
import com.wabit.uecs.pi.webui.WebUIApplication;

/**
 * Raspberry PiのGPIOピンに接続されたアクチュエータの基底実装クラスです。
 * @author WaBit
 *
 * @param <T> 設定値クラス
 */
public abstract class GpioActuatorBase<T extends GpioActuatorConfig> extends PiActuatorBase<T> {

    // 逆転動作させるためのフラグ
    private boolean inverse = false;
    // 手動モード選択ピン
    private GpioPinDigitalInput interlockPin;
    // ロガー
    private Log logger = LogFactory.getLog(getClass());

    /**
     * コンストラクタです。
     * @param id コンポーネントID
     * @param config 設定値
     */
    public GpioActuatorBase(String id, T config) {
        super(id, config);
    }

    /*
     * (非 Javadoc)
     * @see com.wabit.uecs.device.AbstractComponent#onInit()
     */
    @Override
    protected void onInit() throws Exception {
        super.onInit();

        // 基本設定値
        GpioActuatorConfig config = getConfig();
        inverse = config.getBoolean(GpioActuatorConfig.KEY_INVERSE, false);

        // デフォルト値か直前の停止位置で復元する
        setValue(getConfig().getInt(GpioActuatorConfig.KEY_LAST_VALUE,
                getConfig().getInt(GpioActuatorConfig.KEY_DEFAULT_VALUE, 0)));

        // rcM受信時動作
        setAction(ActionMode.rcM, new RcMListenerAction());

        // rcA受信時動作
        setAction(ActionMode.rcA, new RcAListenerAction());

    }

    /**
     * 手動動作モードであるかを判定します。
     * @return 手動動作(ピン状態がLOW)の場合はtrue.
     */
    public boolean isInterlock() {
        if (interlockPin != null && interlockPin.isLow()) {
            return true;
        }
        return false;
    }

    /**
     * 手動操作選択ピンを取得します。
     * @return ピンインスタンス(設定されていない場合はnull)
     */
    public Pin getInterlockPin() {
        GpioActuatorConfig config = getConfig();
        String val = config.getString(GpioActuatorConfig.KEY_GPIO_PIN_INTERLOCK);
        if (val != null && !val.isEmpty()) {
            return ((GpioDeviceBase<?>) getDevice()).getPin(val);
        }
        return null;
    }

    /**
     * 動作反転フラグを取得します。
     * @return 反転の場合はtrue
     */
    boolean isInverse() {
        return this.inverse;
    }

    /**
     * 初期処理を行います。
     */
    @Override
    protected void onStart() throws Exception {

        GpioDeviceBase<?> device = (GpioDeviceBase<?>)getDevice();

        // 本番モードのときだけ実際のGPIO初期化
        if (!WebUIApplication.getNodeInstance().isDevelopmentMode()) {
            // インターロックピンが指定されていた場合の初期設定
            Pin pin = getInterlockPin();
            if (pin != null) {
                // インターロックピンは初期状態でプルアップ(=HIGH)
                // ピンをグランド結線(=LOW)すると、手動操作モードと認識
                GpioPinInput inPin = device.getProvisionedGpioPinInput(pin);
                if (inPin == null) {
                    interlockPin = device.provisionDigitalInputPin(pin, PinPullResistance.PULL_UP);
                } else if (inPin instanceof GpioPinDigitalInput) {
                    interlockPin = (GpioPinDigitalInput)inPin;
                }
                if (interlockPin == null) {
                    throw new Exception("GPIO pin not found : " + pin);
                }
                logger.debug("interlock_pin : " + interlockPin);
                device.addListener(new GpioPinListenerDigital() {

                    @Override
                    public void handleGpioPinDigitalStateChangeEvent(
                            GpioPinDigitalStateChangeEvent event) {
                        if (event.getEventType() != PinEventType.DIGITAL_STATE_CHANGE) {
                            return;
                        }
                        try {
                            if (event.getState().isLow()) {
                                // 手動操作に切り替え
                                changeMode(ActionMode.Interlock);
                            } else {
                                if (getAction(ActionMode.WEB) != null) {
                                    // WEB強制操作に切り替え
                                    changeMode(ActionMode.WEB);
                                } else if (getAction(ActionMode.Autonomy) != null) {
                                    // 自動操作に切り替え
                                    changeMode(ActionMode.Autonomy);
                                } else {
                                    // スタンドアロンに切り替え
                                    changeMode(ActionMode.Standalone);
                                }
                            }
                        } catch (Exception e) {
                            logger.error("mode change error.", e);
                            GpioActuatorBase.this.notifyException(e);
                        }
                    }

                }, interlockPin);
            }
        } else {
            logger.debug("development mode");
        }

        // 初期動作モード設定
        if (isInterlock()) {
            setActionMode(ActionMode.Interlock);
        } else if (getAction(ActionMode.WEB) != null) {
            setActionMode(ActionMode.WEB);
        } else if (getAction(ActionMode.Autonomy) != null) {
            setActionMode(ActionMode.Autonomy);
        } else {
            setActionMode(ActionMode.Standalone);
        }

        super.onStart();
    }

    @Override
    protected void onStop() throws Exception {
        if (interlockPin != null) {
            ((GpioDeviceBase<?>)getDevice()).unprovisionPin(interlockPin);
            interlockPin = null;
        }
        super.onStop();
    }

}
