/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.pi.device.gpio;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.pi4j.io.gpio.GpioPinAnalogInput;
import com.pi4j.io.gpio.event.GpioPinAnalogValueChangeEvent;
import com.pi4j.io.gpio.event.GpioPinListenerAnalog;
import com.pi4j.io.gpio.event.PinEventType;
import com.wabit.uecs.device.actuator.ActuatorComponent;
import com.wabit.uecs.device.actuator.IActuatorAction;

/**
 * スイッチアクチュエータのインターロック動作実装です。
 *
 * @author WaBit
 *
 */
public class AnalogPositioningInterlockAction implements IActuatorAction, GpioPinListenerAnalog {

    private AnalogPositioningActuator<?> positioningActuator;
    private double inputRange;
    // ロガー
    private Log logger = LogFactory.getLog(getClass());

    /**
     * 入力ピンの状態を見て連動させます。
     */
    @Override
    public void execute(ActuatorComponent<?> paramActuatorComponent) throws Exception {
        GpioPinAnalogInput input = positioningActuator.getGpioInputPin();
        if (input != null) {
            int pos = (int) Math.round(input.getValue() / inputRange * 100.0);
            positioningActuator.setPosition(pos);
        }
    }

    /*
     * (非 Javadoc)
     * @see com.wabit.uecs.device.actuator.IActuatorAction#onStart(com.wabit.uecs.device.actuator.ActuatorComponent)
     */
    @Override
    public void onStart(ActuatorComponent<?> paramActuatorComponent) throws Exception {
        if (!(paramActuatorComponent instanceof AnalogPositioningActuator)) {
            throw new IllegalArgumentException("invalid actuator = " + paramActuatorComponent.getClass().getName());
        }
        positioningActuator = (AnalogPositioningActuator<?>) paramActuatorComponent;
        inputRange = positioningActuator.getConfig()
                .getDouble(AnalogPositioningActuatorConfig.KEY_ANALOG_RANGE_IN, 100);
        GpioDeviceBase<?> device = (GpioDeviceBase<?>) positioningActuator.getDevice();
        if (positioningActuator.getGpioInputPin() != null) {
            device.addListener(this, positioningActuator.getGpioInputPin());
        }
        // 初期状態セット
        execute(paramActuatorComponent);
    }

    /*
     * (非 Javadoc)
     * @see com.wabit.uecs.device.actuator.IActuatorAction#onStop(com.wabit.uecs.device.actuator.ActuatorComponent)
     */
    @Override
    public void onStop(ActuatorComponent<?> paramActuatorComponent) throws Exception {

        GpioDeviceBase<?> device = (GpioDeviceBase<?>) positioningActuator.getDevice();
        if (positioningActuator.getGpioInputPin() != null) {
            device.removeListener(this, positioningActuator.getGpioInputPin());
        }
    }

    /**
     * ピン状態変更イベント処理を行います。
     */
    @Override
    public void handleGpioPinAnalogValueChangeEvent(GpioPinAnalogValueChangeEvent event) {
        if (event.getEventType() == PinEventType.ANALOG_VALUE_CHANGE) {
            try {
                execute(positioningActuator);
            } catch (Exception e) {
                logger.error("process faild.", e);
            }
        }

    }
}
