/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.pi.db;

import java.io.InputStream;
import java.io.OutputStream;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map.Entry;
import java.util.concurrent.Callable;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.j256.ormlite.dao.Dao;
import com.j256.ormlite.stmt.DeleteBuilder;
import com.j256.ormlite.stmt.QueryBuilder;
import com.j256.ormlite.stmt.Where;
import com.wabit.uecs.NodeConfig;
import com.wabit.uecs.device.ComponentConfig;
import com.wabit.uecs.device.DeviceConfig;
import com.wabit.uecs.device.IComponent;
import com.wabit.uecs.device.IDevice;
import com.wabit.uecs.pi.AppConstants;
import com.wabit.uecs.pi.device.PiDeviceBase;

/**
 * データベース処理に関する各種ユーティリティメソッドを提供します。
 *
 * @author WaBit
 */
public class DatabaseUtils {
    // データ保存用レコードNo
    private static long recNo;
    private static int logNo;
    private static long idNo;

    // 隠ぺい
    private DatabaseUtils() {
    }

    /**
     * 内部状態を初期化します。
     * @throws SQLException 処理に失敗するとスローされます。
     */
    public static void init() throws SQLException {

        Dao<ComponentValueEntity, Integer> dataDao = DatabaseManager.createDao(ComponentValueEntity.class);
        ComponentValueEntity entity = dataDao.queryBuilder().limit(1L)
                .orderBy(ComponentValueEntity.TIME, false).orderBy(ComponentValueEntity.REC_NO, false).queryForFirst();
        if (entity != null) {
            recNo = entity.getRecNo();
        }
        Dao<LogEntity, Integer> logDao = DatabaseManager.createDao(LogEntity.class);
        QueryBuilder<LogEntity, Integer> query = logDao.queryBuilder();
        LogEntity lastLog = query.limit(1L).orderBy(LogEntity.TIME, false).orderBy(LogEntity.LOG_NO, false)
                .queryForFirst();
        if (lastLog != null) {
            logNo = lastLog.getLogNo();
        }
    }

    /**
     * DBに初期データが登録されているかを調査します。
     * @return 登録されている場合はtrue
     * @throws SQLException 処理に失敗するとスローされます。
     */
    public static boolean isInstalled() throws SQLException {
        // ノード設定が一件も登録されていない場合に未インストールと考える。
        Dao<NodeConfigEntity, String> nodeDao = DatabaseManager.createDao(NodeConfigEntity.class);
        if (nodeDao.countOf() == 0) {
            return false;
        }
        return true;
    }

    /**
     * 設定情報をXML形式で書き出します。
     * @param out 出力先ストリーム
     * @throws Exception 書き込みに失敗した場合。
     */
    public static void writeConfigXml(OutputStream out) throws Exception {
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        DocumentBuilder builder = factory.newDocumentBuilder();
        DOMImplementation domImpl = builder.getDOMImplementation();
        Document document = domImpl.createDocument("", "UecsPiNode", null);
        Element rootElm = document.getDocumentElement();

        Element nodeConfElm = document.createElement("NodeConfig");
        Dao<NodeConfigEntity, String> nodeDao = DatabaseManager.createDao(NodeConfigEntity.class);
        for (NodeConfigEntity entity : nodeDao.queryForAll()) {
            Element entityElm = document.createElement("Entity");
            entityElm.setAttribute("key", entity.getKey());
            entityElm.setAttribute("value", entity.getValue());
            nodeConfElm.appendChild(entityElm);
        }
        rootElm.appendChild(nodeConfElm);

        Element devElm = document.createElement("Device");
        Dao<DeviceEntity, String> deviceDao = DatabaseManager.createDao(DeviceEntity.class);
        for (DeviceEntity entity : deviceDao.queryForAll()) {
            Element entityElm = document.createElement("Entity");
            entityElm.setAttribute("id", entity.getDeviceId());
            entityElm.setAttribute("name", entity.getDeviceName());
            entityElm.setAttribute("javaClass", entity.getDeviceClass());
            devElm.appendChild(entityElm);
        }
        rootElm.appendChild(devElm);

        Element devConfElm = document.createElement("DeviceConfig");
        Dao<DeviceConfigEntity, String> devConfDao = DatabaseManager.createDao(DeviceConfigEntity.class);
        for (DeviceConfigEntity entity : devConfDao.queryForAll()) {
            Element entityElm = document.createElement("Entity");
            entityElm.setAttribute("id", entity.getId());
            entityElm.setAttribute("deviceId", entity.getDeviceId());
            entityElm.setAttribute("key", entity.getKey());
            entityElm.setAttribute("value", entity.getValue());
            devConfElm.appendChild(entityElm);
        }
        rootElm.appendChild(devConfElm);

        Element compoElm = document.createElement("Component");
        Dao<ComponentEntity, String> compoDao = DatabaseManager.createDao(ComponentEntity.class);
        for (ComponentEntity entity : compoDao.queryForAll()) {
            Element entityElm = document.createElement("Entity");
            entityElm.setAttribute("id", entity.getComponentId());
            entityElm.setAttribute("deviceId", entity.getDeviceId());
            entityElm.setAttribute("name", entity.getComponentName());
            entityElm.setAttribute("javaClass", entity.getComponentClass());
            compoElm.appendChild(entityElm);
        }
        rootElm.appendChild(compoElm);

        Element compoConfElm = document.createElement("ComponentConfig");
        Dao<ComponentConfigEntity, String> compoConfDao = DatabaseManager.createDao(ComponentConfigEntity.class);
        for (ComponentConfigEntity entity : compoConfDao.queryForAll()) {
            Element entityElm = document.createElement("Entity");
            entityElm.setAttribute("id", entity.getId());
            entityElm.setAttribute("componentId", entity.getComponentId());
            entityElm.setAttribute("key", entity.getKey());
            entityElm.setAttribute("value", entity.getValue());
            compoConfElm.appendChild(entityElm);
        }
        rootElm.appendChild(compoConfElm);

        TransformerFactory transFactory = TransformerFactory.newInstance();
        Transformer transformer = transFactory.newTransformer();
        DOMSource source = new DOMSource(document);
        StreamResult result = new StreamResult(out);
        transformer.transform(source, result);
    }

    /**
     * XML形式設定情報をDBに読み込みます。
     * 古い設定データはすべて削除されます。
     * @param input
     * @throws Exception
     */
    public static void readConfigXml(InputStream input) throws Exception {
        final Document document = DocumentBuilderFactory
                .newInstance()
                .newDocumentBuilder().parse(input);

        DatabaseManager.callInTransaction(
                new Callable<Void>() {
                    public Void call() throws Exception {
                        Dao<NodeConfigEntity, String> nodeDao = DatabaseManager.createDao(NodeConfigEntity.class);
                        nodeDao.deleteBuilder().delete();
                        NodeList nodeConfElms = document.getElementsByTagName("NodeConfig").item(0).getChildNodes();
                        for (int i = 0; i < nodeConfElms.getLength(); i++) {
                            Node node = nodeConfElms.item(i);
                            if (!(node instanceof Element)) {
                                // 改行等の無視
                                continue;
                            }
                            Element confElm = (Element)node ;
                            NodeConfigEntity entity = new NodeConfigEntity();
                            entity.setKey(confElm.getAttribute("key"));
                            entity.setValue(confElm.getAttribute("value"));
                            nodeDao.create(entity);
                        }

                        Dao<DeviceEntity, String> deviceDao = DatabaseManager.createDao(DeviceEntity.class);
                        deviceDao.deleteBuilder().delete();
                        NodeList devElms = document.getElementsByTagName("Device").item(0).getChildNodes();
                        for (int i = 0; i < devElms.getLength(); i++) {
                            Node node = devElms.item(i);
                            if (!(node instanceof Element)) {
                                // 改行等の無視
                                continue;
                            }
                            Element devElm = (Element)node;
                            DeviceEntity entity = new DeviceEntity();
                            entity.setDeviceId(devElm.getAttribute("id"));
                            entity.setDeviceName(devElm.getAttribute("name"));
                            entity.setDeviceClass(devElm.getAttribute("javaClass"));
                            deviceDao.create(entity);
                        }

                        Dao<DeviceConfigEntity, String> devConfDao = DatabaseManager
                                .createDao(DeviceConfigEntity.class);
                        devConfDao.deleteBuilder().delete();
                        NodeList devConfElms = document.getElementsByTagName("DeviceConfig").item(0).getChildNodes();
                        for (int i = 0; i < devConfElms.getLength(); i++) {
                            Node node = devConfElms.item(i);
                            if (!(node instanceof Element)) {
                                // 改行等の無視
                                continue;
                            }
                            Element devConfElm = (Element)node;
                            DeviceConfigEntity entity = new DeviceConfigEntity();
                            entity.setId(devConfElm.getAttribute("id"));
                            entity.setDeviceId(devConfElm.getAttribute("deviceId"));
                            entity.setKey(devConfElm.getAttribute("key"));
                            entity.setValue(devConfElm.getAttribute("value"));
                            devConfDao.create(entity);
                        }

                        Dao<ComponentEntity, String> compoDao = DatabaseManager.createDao(ComponentEntity.class);
                        compoDao.deleteBuilder().delete();
                        NodeList compoElms = document.getElementsByTagName("Component").item(0).getChildNodes();
                        for (int i = 0; i < compoElms.getLength(); i++) {
                            Node node = compoElms.item(i);
                            if (!(node instanceof Element)) {
                                // 改行等の無視
                                continue;
                            }
                            Element compoElm = (Element) node;
                            ComponentEntity entity = new ComponentEntity();
                            entity.setComponentId(compoElm.getAttribute("id"));
                            entity.setDeviceId(compoElm.getAttribute("deviceId"));
                            entity.setComponentName(compoElm.getAttribute("name"));
                            entity.setComponentClass(compoElm.getAttribute("javaClass"));
                            compoDao.create(entity);
                        }

                        Dao<ComponentConfigEntity, String> compoConfDao = DatabaseManager
                                .createDao(ComponentConfigEntity.class);
                        compoConfDao.deleteBuilder().delete();
                        NodeList compoConfElms = document.getElementsByTagName("ComponentConfig").item(0)
                                .getChildNodes();
                        for (int i = 0; i < compoConfElms.getLength(); i++) {
                            Node node = compoConfElms.item(i);
                            if (!(node instanceof Element)) {
                                // 改行等の無視
                                continue;
                            }
                            Element compoConfElm = (Element)node;
                            ComponentConfigEntity entity = new ComponentConfigEntity();
                            entity.setId(compoConfElm.getAttribute("id"));
                            entity.setComponentId(compoConfElm.getAttribute("componentId"));
                            entity.setKey(compoConfElm.getAttribute("key"));
                            entity.setValue(compoConfElm.getAttribute("value"));
                            compoConfDao.create(entity);
                        }
                        return null;
                    }
                });
    }

    /**
     * DBからノード設定をロードします。
     * @return ノード設定
     * @throws SQLException 処理に失敗するとスローされます。
     */
    public static NodeConfig loadNodeConfig() throws SQLException {
        Dao<NodeConfigEntity, String> nodeDao = DatabaseManager.createDao(NodeConfigEntity.class);
        NodeConfig nodeConf = new NodeConfig();
        for (NodeConfigEntity entity : nodeDao.queryForAll()) {
            nodeConf.setString(entity.getKey(), entity.getValue());
        }
        return nodeConf;
    }

    /**
     * DBからデバイス情報をロードします。
     * @return デバイスリスト
     * @throws Exception 処理に失敗するとスローされます。
     */
    public static List<PiDeviceBase<?>> loadDevices() throws Exception {
        Dao<DeviceEntity, String> deviceDao = DatabaseManager.createDao(DeviceEntity.class);
        Dao<DeviceConfigEntity, String> confDao = DatabaseManager.createDao(DeviceConfigEntity.class);
        List<DeviceEntity> list = deviceDao.queryBuilder().orderBy(DeviceEntity.DEVICE_ID, true).query();
        List<PiDeviceBase<?>> deviceList = new ArrayList<PiDeviceBase<?>>();
        for (DeviceEntity entity : list) {
            String clazz = entity.getDeviceClass();
            PiDeviceBase<?> device = (PiDeviceBase<?>) Class.forName(clazz).getConstructor(String.class)
                    .newInstance(entity.getDeviceId());
            device.getConfig().setString(DeviceConfig.KEY_DEVICE_NAME, entity.getDeviceName());
            // DBから設定値を読込
            for (DeviceConfigEntity cnf : confDao.queryForEq(DeviceConfigEntity.DEVICE_ID, entity.getDeviceId())) {
                device.getConfig().setString(cnf.getKey(), cnf.getValue());
            }
            deviceList.add(device);
        }

        return deviceList;
    }

    /**
     * DBからコンポーネント情報をロードします。
     * @param deviceId デバイスID
     * @return コンポーネントリスト
     * @throws Exception 処理に失敗するとスローされます。
     */
    public static List<IComponent<?>> loadComponents(String deviceId) throws Exception {

        // ひもつくコンポーネント生成
        Dao<ComponentEntity, String> compoDao = DatabaseManager.createDao(ComponentEntity.class);
        Dao<ComponentConfigEntity, String> compoConfDao = DatabaseManager.createDao(ComponentConfigEntity.class);
        List<IComponent<?>> components = new ArrayList<IComponent<?>>();
        for (ComponentEntity entity : compoDao.queryBuilder().orderBy(ComponentEntity.COMPONENT_ID, true).where()
                .eq(ComponentEntity.DEVICE_ID, deviceId).query()) {
            String clazz = entity.getComponentClass();
            String compoId = entity.getComponentId();
            IComponent<?> compo = (IComponent<?>) Class.forName(clazz).getConstructor(String.class)
                    .newInstance(compoId);
            compo.getConfig().setString(ComponentConfig.KEY_COMPONENT_NAME, entity.getComponentName());
            for (ComponentConfigEntity cnf_entity : compoConfDao
                    .queryForEq(ComponentConfigEntity.COMPONENT_ID, compoId)) {
                compo.getConfig().setString(cnf_entity.getKey(), cnf_entity.getValue());
            }
            components.add(compo);
        }
        return components;
    }

    /**
     * デバイス情報（DeviceConfigや内包するコンポーネントも含む）をDBに保存します。
     * @param device デバイスインスタンス
     * @throws SQLException 処理に失敗するとスローされます。
     */
    public static void saveDevice(IDevice<?> device) throws SQLException {

        deleteDevice(device.getId());

        Dao<DeviceEntity, String> devDao = DatabaseManager.createDao(DeviceEntity.class);

        DeviceEntity devEnt = new DeviceEntity();
        devEnt.setDeviceClass(device.getClass().getName());
        devEnt.setDeviceId(device.getId());
        devEnt.setDeviceName(device.getConfig().getString(DeviceConfig.KEY_DEVICE_NAME));
        devDao.createOrUpdate(devEnt);

        saveDeviceConfig(device.getId(), device.getConfig());
        for (IComponent<?> compo : device.listComponents()) {
            saveComponent(device.getId(), compo);
        }
    }

    /**
     * コンポーネント情報（ComponentConfigも含む）をDBに保存します。
     * @param deviceId デバイスID
     * @param component コンポーネントインスタンス
     * @throws SQLException 処理に失敗するとスローされます。
     */
    public static void saveComponent(String deviceId, IComponent<?> component) throws SQLException {

        deleteComponent(component.getId());

        Dao<ComponentEntity, String> devDao = DatabaseManager.createDao(ComponentEntity.class);

        ComponentEntity compoEnt = new ComponentEntity();
        compoEnt.setComponentClass(component.getClass().getName());
        compoEnt.setDeviceId(deviceId);
        compoEnt.setComponentId(component.getId());
        compoEnt.setComponentName(component.getConfig().getString(ComponentConfig.KEY_COMPONENT_NAME));
        devDao.createOrUpdate(compoEnt);

        saveComponentConfig(component.getId(), component.getConfig());
    }

    /**
     * デバイス設定を保存するためのユーティリティメソッドです。
     * @param deviceId デバイスID
     * @param conf 設定値オブジェクト
     * @throws SQLException 処理に失敗するとスローされます。
     */
    public static void saveDeviceConfig(String deviceId, DeviceConfig conf) throws SQLException {
        Dao<DeviceConfigEntity, String> confDao = DatabaseManager.createDao(DeviceConfigEntity.class);

        // 古い設定値は全削除してから書き込む
        DeleteBuilder<DeviceConfigEntity, String> del = confDao.deleteBuilder();
        del.where().eq(DeviceConfigEntity.DEVICE_ID, deviceId);
        confDao.delete(del.prepare());

        for (Entry<Object, Object> entry : conf.entrySet()) {
            DeviceConfigEntity confEnt = new DeviceConfigEntity();
            confEnt.setId(deviceId + "." + entry.getKey());
            confEnt.setDeviceId(deviceId);
            confEnt.setKey(entry.getKey().toString());
            if (entry.getValue() != null) {
                confEnt.setValue(entry.getValue().toString());
            }
            confDao.createOrUpdate(confEnt);
        }

    }

    /**
     * ノード設定を保存するためのユーティリティメソッドです。
     * @param conf 設定値オブジェクト
     * @throws SQLException 処理に失敗するとスローされます。
     */
    public static void saveNodeConfig(NodeConfig conf) throws SQLException {
        Dao<NodeConfigEntity, String> dao = DatabaseManager.createDao(NodeConfigEntity.class);

        // 古い設定値は全削除してから書き込む
        dao.deleteBuilder().delete();

        for (Entry<Object, Object> entry : conf.entrySet()) {
            NodeConfigEntity entity = new NodeConfigEntity();
            entity.setKey(entry.getKey().toString());
            if (entry.getValue() != null) {
                entity.setValue(entry.getValue().toString());
            }
            dao.createOrUpdate(entity);
        }

    }

    /**
     * ノード設定１項目を保存するためのユーティリティメソッドです。
     * @param key 設定値キー
     * @param param 設定値
     * @throws SQLException 処理に失敗するとスローされます。
     */
    public static void saveNodeConfig(String key, String param) throws SQLException {
        Dao<NodeConfigEntity, String> dao = DatabaseManager.createDao(NodeConfigEntity.class);
        NodeConfigEntity entity = new NodeConfigEntity();
        entity.setKey(key);
        entity.setValue(param);
        dao.createOrUpdate(entity);

    }

    /**
     * コンポーネント設定を保存するためのユーティリティメソッドです。
     * @param componentId コンポーネントID
     * @param conf 設定値オブジェクト
     * @throws SQLException 処理に失敗するとスローされます。
     */
    public static void saveComponentConfig(String componentId, ComponentConfig conf) throws SQLException {

        Dao<ComponentConfigEntity, String> confDao = DatabaseManager.createDao(ComponentConfigEntity.class);
        // 最初に古い設定値を全削除
        DeleteBuilder<ComponentConfigEntity, String> del = confDao.deleteBuilder();
        del.where().eq(ComponentConfigEntity.COMPONENT_ID, componentId);
        confDao.delete(del.prepare());

        for (Entry<Object, Object> entry : conf.entrySet()) {
            ComponentConfigEntity confEnt = new ComponentConfigEntity();
            confEnt.setId(componentId + "." + entry.getKey().toString());
            confEnt.setComponentId(componentId);
            confEnt.setKey(entry.getKey().toString());
            if (entry.getValue() != null) {
                confEnt.setValue(entry.getValue().toString());
            }
            confDao.create(confEnt);
        }

    }

    /**
     * コンポーネント設定１項目を保存するためのユーティリティメソッドです。
     * @param componentId コンポーネントID
     * @param key 設定値キー
     * @param param 設定値
     * @throws SQLException 処理に失敗するとスローされます。
     */
    public static void saveComponentConfig(String componentId, String key, String param) throws SQLException {

        Dao<ComponentConfigEntity, String> confDao = DatabaseManager.createDao(ComponentConfigEntity.class);

        ComponentConfigEntity confEnt = new ComponentConfigEntity();
        confEnt.setId(componentId + "." + key);
        confEnt.setComponentId(componentId);
        confEnt.setKey(key);
        confEnt.setValue(param);
        confDao.createOrUpdate(confEnt);

    }

    /**
     * コンポーネントデータ値を保存します。
     * @param componentId コンポーネントID
     * @param time 記録日時
     * @param value データ値
     * @throws SQLException 処理に失敗するとスローされます。
     */
    public static void saveComponentData(String componentId, long time, Number value) throws SQLException {
        if (componentId == null || value == null) {
            return;
        }
        Dao<ComponentValueEntity, Integer> dataDao = DatabaseManager.createDao(ComponentValueEntity.class);
        ComponentValueEntity entity = new ComponentValueEntity();
        entity.setRecNo(nextRecNo());
        entity.setComponentId(componentId);
        entity.setValue(value.doubleValue());
        entity.setTime(time);
        dataDao.createOrUpdate(entity);
    }

    /**
     * デバイスを削除します。
     * @param deviceId 削除対象のデバイスID
     * @throws SQLException 処理に失敗するとスローされます。
     */
    public static void deleteDevice(String deviceId) throws SQLException {
        Dao<DeviceEntity, String> devDao = DatabaseManager.createDao(DeviceEntity.class);
        List<DeviceEntity> devList = devDao.queryForEq(DeviceEntity.DEVICE_ID, deviceId);
        if (devList.size() == 0) {
            return;
        }

        Dao<ComponentEntity, String> compoDao = DatabaseManager.createDao(ComponentEntity.class);
        Dao<ComponentConfigEntity, String> confDao = DatabaseManager.createDao(ComponentConfigEntity.class);
        Dao<ComponentValueEntity, Integer> dataDao = DatabaseManager.createDao(ComponentValueEntity.class);
        for (ComponentEntity compo : compoDao.queryForEq(ComponentEntity.DEVICE_ID, deviceId)) {
            String componentId = compo.getComponentId();
            DeleteBuilder<ComponentConfigEntity, String> del = confDao.deleteBuilder();
            del.where().eq(ComponentConfigEntity.COMPONENT_ID, componentId);
            confDao.delete(del.prepare());
            DeleteBuilder<ComponentValueEntity, Integer> ddel = dataDao.deleteBuilder();
            ddel.where().eq(ComponentValueEntity.COMPONENT_ID, componentId);
            dataDao.delete(ddel.prepare());

            compoDao.deleteById(componentId);
        }

        Dao<DeviceConfigEntity, String> dconfDao = DatabaseManager.createDao(DeviceConfigEntity.class);
        DeleteBuilder<DeviceConfigEntity, String> del = dconfDao.deleteBuilder();
        del.where().eq(DeviceConfigEntity.DEVICE_ID, deviceId);
        dconfDao.delete(del.prepare());

        devDao.deleteById(deviceId);

    }

    /**
     * コンポーネントを削除します。
     * @param componentId 削除対象のコンポーネントID
     * @throws SQLException 処理に失敗するとスローされます。
     */
    public static void deleteComponent(String componentId) throws SQLException {
        Dao<ComponentConfigEntity, String> confDao = DatabaseManager.createDao(ComponentConfigEntity.class);
        DeleteBuilder<ComponentConfigEntity, String> del = confDao.deleteBuilder();
        del.where().eq(ComponentConfigEntity.COMPONENT_ID, componentId);
        confDao.delete(del.prepare());

        Dao<ComponentValueEntity, Integer> dataDao = DatabaseManager.createDao(ComponentValueEntity.class);
        DeleteBuilder<ComponentValueEntity, Integer> ddel = dataDao.deleteBuilder();
        ddel.where().eq(ComponentValueEntity.COMPONENT_ID, componentId);
        dataDao.delete(ddel.prepare());

        Dao<ComponentEntity, String> compoDao = DatabaseManager.createDao(ComponentEntity.class);
        compoDao.deleteById(componentId);

    }

    /**
     * 時間範囲を指定してコンポーネント値リストを取得します。
     * @param componentId コンポーネントID
     * @param start 開始日時
     * @param end 終了日時
     * @return データ値リスト
     * @throws SQLException 処理に失敗するとスローされます。
     */
    public static List<ComponentValueEntity> getComponentValues(String componentId, Date start, Date end)
            throws SQLException {
        Dao<ComponentValueEntity, Integer> dataDao = DatabaseManager.createDao(ComponentValueEntity.class);
        QueryBuilder<ComponentValueEntity, Integer> query = dataDao.queryBuilder();
        Where<ComponentValueEntity, Integer> where = null;
        if (componentId != null) {
            where = query.where().eq(ComponentValueEntity.COMPONENT_ID, componentId);
        }
        if (start != null) {
            if (where != null) {
                where = where.and().ge(ComponentValueEntity.TIME, start.getTime());
            } else {
                where = query.where().ge(ComponentValueEntity.TIME, start.getTime());
            }
        }
        if (end != null) {
            if (where != null) {
                where = where.and().le(ComponentValueEntity.TIME, end.getTime());
            } else {
                where = query.where().le(ComponentValueEntity.TIME, end.getTime());
            }
        }

        return query.query();
    }

    /**
     * コンポーネント値データをすべて削除します。
     * @return 削除レコード数
     * @throws SQLException 処理に失敗するとスローされます。
     */
    public static int clearComponentValues() throws SQLException {
        Dao<ComponentValueEntity, Integer> dataDao = DatabaseManager.createDao(ComponentValueEntity.class);
        int num = dataDao.deleteBuilder().delete();
        recNo = 0;
        return num;
    }


    /**
     * ログリストを取得します。
     * @param fetchSize 最大取得数
     * @return ログリスト
     * @throws SQLException 処理に失敗するとスローされます。
     */
    public static List<LogEntity> getLogs(int fetchSize) throws SQLException {
        Dao<LogEntity, Integer> logDao = DatabaseManager.createDao(LogEntity.class);
        QueryBuilder<LogEntity, Integer> query = logDao.queryBuilder().limit((long) fetchSize)
                .orderBy(LogEntity.TIME, false).orderBy(LogEntity.LOG_NO, false);
        return query.query();

    }

    /**
     * カテゴリを指定してログリストを取得します。
     * @param category カテゴリ
     * @param fetchSize 最大取得数
     * @return ログリスト
     * @throws SQLException 処理に失敗するとスローされます。
     */
    public static List<LogEntity> getLogs(String category, int fetchSize) throws SQLException {
        Dao<LogEntity, Integer> logDao = DatabaseManager.createDao(LogEntity.class);
        QueryBuilder<LogEntity, Integer> query = logDao.queryBuilder().limit((long) fetchSize)
                .orderBy(LogEntity.TIME, false).orderBy(LogEntity.LOG_NO, false);
        return query.where().eq(LogEntity.CATEGORY, category).query();
    }

    /**
     * ログ情報を保存します。
     * @param entity ログ情報エンティティ
     * @throws SQLException 処理に失敗するとスローされます。
     */
    public static void saveLog(LogEntity entity) throws SQLException {
        Dao<LogEntity, Integer> logDao = DatabaseManager.createDao(LogEntity.class);
        entity.setLogNo(nextLogNo());
        logDao.createOrUpdate(entity);
    }

    /**
     * ログ情報をすべて削除します。
     * @return 削除レコード数
     * @throws SQLException 処理に失敗するとスローされます。
     */
    public static int clearLogs() throws SQLException {
        Dao<LogEntity, Integer> logDao = DatabaseManager.createDao(LogEntity.class);
        int num = logDao.deleteBuilder().delete();
        logNo = 0;
        return num;
    }

    /**
     * 新規デバイスIDを作成します。
     * @return デバイスID
     */
    public static synchronized String nextDeviceId() {
        return "D:" + System.currentTimeMillis() + "-" + idNo++;
    }

    /**
     * 新規コンポーネントIDを作成します。
     * @return コンポーネントID
     */
    public static synchronized String nextComponentId() {
        return "C:" + System.currentTimeMillis() + "-" + idNo++;
    }

    /**
     * データ保存レコード番号を取得します。
     * @return レコードNo
     */
    private static synchronized long nextRecNo() {
        if (recNo >= AppConstants.MAX_DATA_NUM) {
            recNo = 0;
        }
        return ++recNo;
    }

    /**
     * ログ保存用レコード番号を取得します。
     * @return ログNo
     */
    private static synchronized int nextLogNo() {
        if (logNo >= AppConstants.MAX_LOG_NUM) {
            logNo = 0;
        }
        logNo++;
        return logNo;
    }

}
