/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.pi;

import java.sql.Driver;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Enumeration;
import java.util.List;
import java.util.Locale;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.joda.time.DateTimeZone;

import com.wabit.uecs.AbstractUecsNode;
import com.wabit.uecs.Ccm;
import com.wabit.uecs.CcmService;
import com.wabit.uecs.ICcmServiceListener;
import com.wabit.uecs.IUecsNode;
import com.wabit.uecs.IUecsNodeListener;
import com.wabit.uecs.NodeRuntimeException;
import com.wabit.uecs.UecsConstants;
import com.wabit.uecs.ccm.DateCcm;
import com.wabit.uecs.ccm.TimeCcm;
import com.wabit.uecs.device.IDevice;
import com.wabit.uecs.pi.db.DatabaseUtils;
import com.wabit.uecs.pi.device.PiDeviceBase;
import com.wabit.uecs.pi.util.MessageCode;
import com.wabit.uecs.protocol.handler.DataProtocolHandler;
import com.wabit.uecs.protocol.handler.ScanProtocolHandler;
import com.wabit.uecs.protocol.handler.SearchProtocolHandler;

/**
 * Raspberry Pi 上で動作するUECSノード実装です。
 * UECS Ver1.00-E10準拠の基本プロトコルが動作します。
 *
 * @author WaBit
 */
public class UecsPiNode extends AbstractUecsNode<UecsPiNodeConfig> {

    // 標準ロガー
    private Log logger = LogFactory.getLog(getClass());
    // 開発モードフラグ
    private boolean isDevelopmentMode;
    // 日時補正フラグ
    private boolean receiveDateTime;
    private boolean timeAdjusted;



    /**
     * デフォルトコンストラクタ。
     */
    public UecsPiNode() {
        this(new UecsPiNodeConfig());
    }

    /**
     * 初期設定値を変更可能なコンストラクタです。
     * @param config 設定値
     */
    public UecsPiNode(UecsPiNodeConfig config) {
        super(config);
        // UECS ver1.0 動作仕様プロトコル登録
        addProtocol(new DataProtocolHandler(this));
        addProtocol(new SearchProtocolHandler(this));
        addProtocol(new ScanProtocolHandler(this));
        addListener(new UecsPiNodeListener());
    }

    /**
     * 開発モードフラグを取得します。
     * @return 開発モードの場合はtrue
     */
    public boolean isDevelopmentMode() {
        return isDevelopmentMode;
    }

    /**
     * 開発モードフラグを設定します
     * @param isDevelopmentMode フラグ
     */
    public void setDevelopmentMode(boolean isDevelopmentMode) {
        this.isDevelopmentMode = isDevelopmentMode;
    }

    /**
     * デフォルトUECSバージョンを返します。
     */
    @Override
    public String getUecsVersion() {
        return UecsConstants.DEFAULT_VERSION;
    }

    /**
     * 内部でDBが初期化処理が実行されます。
     */
    @Override
    public synchronized void setup() {

        // ロケールの設定
        String locale = getConfig().getString(UecsPiNodeConfig.KEY_LOCALE);
        if (locale != null && locale.length() > 0) {
            Locale.setDefault(Locale.forLanguageTag(locale));
            logger.debug("set Locale = " + Locale.getDefault().toLanguageTag());
        }

        // タイムゾーンの設定
        String timezone = getConfig().getString(UecsPiNodeConfig.KEY_TIME_ZONE);
        if (timezone != null && timezone.length() > 0) {
            DateTimeZone.setDefault(DateTimeZone.forID(timezone));
            logger.debug("set TimeZone = " + DateTimeZone.getDefault().getID());
        }

        super.setup();
        try {
            // DB接続ユーティリティを初期化
            DatabaseUtils.init();
        } catch (SQLException e) {
            onStatus(UecsConstants.STATUS_ALART_INIT_FAILED);
            notifyException(e);
            throw new NodeRuntimeException(e);
        }

    }

    /**
     * 内部で動作インジケータの初期化が実行されます。
     * ただし、開発モード(DevelopmentMode)フラグがtrueの場合はインジケータは動作しません。
     */
    @Override
    public synchronized void start() {
        try {

            // DATE, TIME CCM送信
            if (getConfig().getBoolean(UecsPiNodeConfig.KEY_SEND_DATETIME, false)) {
                DateCcm dateCcm = new DateCcm();
                dateCcm.setSide(UecsConstants.SENDER);
                CcmService ccmSv = new CcmService(dateCcm);
                ccmSv.setName(getName() + " [Date]");
                ccmSv.setValue(System.currentTimeMillis(), dateCcm.getNumberValue());
                addCcmService(ccmSv);

                TimeCcm timeCcm = new TimeCcm();
                timeCcm.setSide(UecsConstants.SENDER);
                ccmSv = new CcmService(timeCcm);
                ccmSv.setName(getName() + " [Time]");
                ccmSv.setValue(System.currentTimeMillis(), timeCcm.getNumberValue());
                addCcmService(ccmSv);
            }

            // DATE, TIME CCM受信
            if (getConfig().getBoolean(UecsPiNodeConfig.KEY_RECEIVE_DATETIME, false)) {
                DateTimeCcmListener dtLsn = new DateTimeCcmListener();
                DateCcm dateCcm = new DateCcm();
                dateCcm.setSide(UecsConstants.RECEIVER);
                CcmService ccmSv = new CcmService(dateCcm);
                ccmSv.setName(getName() + " [Date]");
                ccmSv.addListener(dtLsn);
                addCcmService(ccmSv);
                TimeCcm timeCcm = new TimeCcm();
                timeCcm.setSide(UecsConstants.RECEIVER);
                ccmSv = new CcmService(timeCcm);
                ccmSv.setName(getName() + " [Time]");
                ccmSv.addListener(dtLsn);
                addCcmService(ccmSv);
                receiveDateTime = true;
            } else {
                receiveDateTime = false;
            }

            // 時刻補正が完了するまでデバイス動作させないための処理
            if (receiveDateTime && !timeAdjusted) {
                onStatus(AppConstants.STATUS_ATTENTION_DATETIME);
                for (PiDeviceBase<?> dev : listDevices(PiDeviceBase.class)) {
                    dev.setEnabled(false);
                }
            }
            super.start();
        } catch (Error e) {
            onStatus(UecsConstants.STATUS_ALART_INIT_FAILED);
            notifyException(new NodeRuntimeException(e));
            throw e;
        }

    }

    /**
     * 内部でインジケータ動作の停止処理が行われます。
     * ただし、開発モード(DevelopmentMode)フラグがtrueの場合はインジケータは動作しません。
     */
    @Override
    public synchronized void stop() {
        super.stop();
    }


    /**
     * 最終停止処理(GPIO動作停止などアプリ終了時に行う処理)を行います。
     */
    public synchronized void onDestroy() {
    }

    /**
     * オブジェクト破棄時にJDBCドライバを削除します。
     */
    @Override
    protected void finalize() throws Throwable {
        Enumeration<Driver> drivers = DriverManager.getDrivers();
        while (drivers.hasMoreElements()) {
            Driver driver = drivers.nextElement();
            try {
                DriverManager.deregisterDriver(driver);
                logger.info(String.format("deregistering jdbc driver: %s", driver));
            } catch (SQLException e) {
                logger.error(String.format("Error deregistering driver %s", driver), e);
            }
        }
        super.finalize();
    }

    /**
     * 非同期でノードを遅延再起動します。
     * @param delay 実行待ち時間(msec)
     * @throws Exception 再起動に失敗するとスローされます。
     */
    public synchronized void restartAsync(long delay) throws Exception {

        if (!DatabaseUtils.isInstalled()) {
            // 初期データ登録
            DatabaseUtils.saveNodeConfig(getConfig());
            installProcess();
        }

        getConfig().putAll(DatabaseUtils.loadNodeConfig());
        List<PiDeviceBase<?>> devices = DatabaseUtils.loadDevices();

        Thread th = new Thread(new RestartTask(devices, delay));
        th.start();
    }

    /**
     * DBから各種設定を読み込み、ノードを再起動します。
     *
     * @throws Exception 再起動に失敗するとスローされます。
     */
    public synchronized void restart() throws Exception {
        logger.debug("restart node.");
        //起動中であれば、停止
        if (isActive()) {
            stop();
        }

        if (!DatabaseUtils.isInstalled()) {
            // 初期データ登録
            DatabaseUtils.saveNodeConfig(getConfig());
            installProcess();
        }

        getConfig().putAll(DatabaseUtils.loadNodeConfig());
        setup();

        //デバイス登録
        List<PiDeviceBase<?>> devices = DatabaseUtils.loadDevices();
        for (PiDeviceBase<?> device : devices) {
            addDevice(device);
        }

        // ノード起動
        start();
    }


    /**
     * インストール直後の初期データ登録処理を記述します。
     * デバイスを初期追加登録したい場合は、継承クラスでオーバーライドしてください。
     */
    protected void installProcess() throws Exception {

    }


    /*
     * 内部ノード動作リスナー
     */
    private class UecsPiNodeListener implements IUecsNodeListener {

        @Override
        public void nodeStarted(IUecsNode<?> paramIUecsNode) {
            logger.info("node started.");
            UecsPiLogger.log(UecsPiLogger.CATEGORY_NODE, MessageCode.NODE_STARTED);
        }

        @Override
        public void nodeStopped(IUecsNode<?> paramIUecsNode) {
            logger.info("node stopped.");
            UecsPiLogger.log(UecsPiLogger.CATEGORY_NODE, MessageCode.NODE_STOPPED);
        }

        @Override
        public void handleNodeException(IUecsNode<?> node, Exception e) {
            logger.error("node error.", e);
            UecsPiLogger.log(UecsPiLogger.CATEGORY_NODE, MessageCode.ALERT, e.getMessage());
        }

    }

    /*
     * 非同期リスタート用処理
     */
    private class RestartTask implements Runnable {
        private List<PiDeviceBase<?>> svlist;

        private long delay;

        public RestartTask(List<PiDeviceBase<?>> devs, long delay) {
            this.svlist = devs;
            this.delay = delay;
        }

        @Override
        public void run() {
            try {
                Thread.sleep(delay);

                logger.debug("restart node.");
                //起動中であれば、停止
                if (isActive()) {
                    stop();
                }

                setup();

                //デバイス登録
                for (IDevice<?> device : svlist) {
                    addDevice(device);
                }

                // ノード起動
                start();
            } catch (Exception e) {
                logger.error(e);
            }

        }
    }

    // 日時補正リスナー
    private class DateTimeCcmListener implements ICcmServiceListener {
        private CcmService date;
        private CcmService time;

        @Override
        public void ccmSent(CcmService source, Ccm value) {
        }

        @Override
        public void ccmReceived(CcmService source, Ccm value) {
        }

        @Override
        public void ccmValueChanged(CcmService source, Ccm value) {
            if (date == null && value instanceof DateCcm) {
                date = source;
            } else if (time == null && value instanceof TimeCcm) {
                time = source;
            }
            if (value instanceof TimeCcm && date != null && !date.isExpired()) {
                int ymd = date.getCcm().getNumberValue().intValue();
                int hms = value.getNumberValue().intValue();
                // 日付変更付近5分以内は、Dateが未更新だと大幅にずれる可能性があるので、見送る
                if (hms <= 500) {
                    return;
                }
                try {
                    SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMddHHmmss");
                    String dtstr = String.format("20%6d", ymd) + String.format("%6d", hms);
                    Date rdate = sdf.parse(dtstr);
                    if (!timeAdjusted || Math.abs(System.currentTimeMillis() - rdate.getTime()) > 60000L) {
                        // UECS運用ガイドラインに従い、60秒以上誤差があれば補正する
                        if (!isDevelopmentMode()) {
                            Runtime rt = Runtime.getRuntime();
                            // 指定されたフォーマットのオブジェクト生成
                            sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                            rt.exec(new String[] { "sh", AppConstants.SCRIPT_CHANGE_DATETIME,
                                    sdf.format(date) });
                        }
                        timeAdjusted = true;
                        logger.info("DateTime adjusted. value=" + dtstr);
                        UecsPiLogger.log(UecsPiLogger.CATEGORY_NODE, MessageCode.DATETIME_ADJUSTED);
                        // 時刻補正されたので、再起動する
                        restartAsync(1000L);
                    }
                } catch (Exception e) {
                    logger.error("DateTime adjusting error.", e);
                }
            }
        }

        @Override
        public void ccmExpired(CcmService source, Ccm value) {
        }

    }

}
