/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.pi.util;

import java.io.UnsupportedEncodingException;

/**
 * バイナリ値の各種変換ユーティリティクラスです。
 *
 * @author WaBit
 *
 */
public class BytesUtils {

    /**
     * HEX値を出力します。
     * @param bytes
     */
    public static void printHex(byte[] bytes) {
        for (byte b : bytes) {
            System.out.printf("'%s'(%02X) ", (char) b, b);
        }
        System.out.println("]");
    }


    /**
     * バイト配列から16進数文字列を取得します。
     * @param bytes
     */
    public static String toHexString(byte[] bytes) {
        StringBuilder sb = new StringBuilder();
        for (byte b : bytes) {
            sb.append(String.format("%02X", b));
        }
        return sb.toString();
    }

    /**
     * バイト配列から16進数文字列を取得します。
     * @param bytes
     * @param splitter 分割記号
     */
    public static String toHexString(byte[] bytes, String splitter) {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < bytes.length; i++) {
            if (i != 0) {
                sb.append(splitter);
            }
            sb.append(String.format("%02X", bytes[i]));
        }
        return sb.toString();
    }

    /**
     * 16進数文字列からバイト配列を取得します。
     * @param hex
     */
    public static byte[] toHexBytes(String hex) {
        if (hex == null || hex.length() % 2 != 0) {
            throw new IllegalArgumentException("illegal string : " + hex);
        }
        byte[] bytes = new byte[hex.length() / 2];
        for (int index = 0; index < bytes.length; index++) {
            bytes[index] = (byte) Integer.parseInt(hex.substring(index * 2, (index + 1) * 2), 16);
        }
        return bytes;
    }

    /**
     * バイト配列から文字列に変換します。
     * @param bytes
     * @param charset
     * @return 変換後文字列
     */
    public static String toString(byte[] bytes, String charset) {
        if (bytes == null) {
            return null;
        }
        try {
            return new String(bytes, charset);
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * バイト配列から文字列に変換します。
     * @param bytes
     * @param offset
     * @param length
     * @param charset
     * @return 変換後文字列
     */
    public static String toString(byte[] bytes, int offset, int length, String charset) {
        if (bytes == null) {
            return null;
        }
        try {
            return new String(bytes, offset, length, charset);
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 文字列からバイト配列に変換します。
     * @param val
     * @param charset
     * @return バイト配列
     */
    public static byte[] toBytes(String val, String charset) {
        if (val == null) {
            return null;
        }
        try {
            return val.getBytes(charset);
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
    }


}
