/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.pi.device.i2c;

import java.io.IOException;

import com.pi4j.io.i2c.I2CDevice;

/**
 * I2C通信用各種ユーティリティメソッドを提供します。
 * @author WaBit
 *
 */
public class I2cUtils {

    // 隠ぺい
    private I2cUtils() {
    }

    /**
     * 1byteのデータ書き込み処理を行います。
     * @param i2cDevice 出力対象デバイス
     * @param tryCount リトライ回数
     * @param b 出力データ
     * @throws IOException 処理が失敗した場合にスローされます。
     */
    public static void write(I2CDevice i2cDevice, int tryCount, byte b) throws IOException {
        IOException ex = null;
        while (tryCount > 0) {
            try {
                i2cDevice.write(b);
                return;
            } catch (IOException e) {
                ex = e;
            }
            tryCount--;
        }
        if (ex != null) {
            throw ex;
        }
    }

    /**
     * 連続したデータの書き込み処理を行います。
     * @param i2cDevice 出力対象デバイス
     * @param tryCount リトライ回数
     * @param buffer 出力バッファー
     * @param offset 開始位置
     * @param size 出力サイズ
     * @throws IOException 処理が失敗した場合にスローされます。
     */
    public static void write(I2CDevice i2cDevice, int tryCount, byte[] buffer, int offset, int size)
            throws IOException {
        IOException ex = null;
        while (tryCount > 0) {
            try {
                i2cDevice.write(buffer, offset, size);
                return;
            } catch (IOException e) {
                ex = e;
            }
            tryCount--;
        }
        if (ex != null) {
            throw ex;
        }
    }

    /**
     * データの読み込み処理を行います。
     * @param i2cDevice 入力対象デバイス
     * @param waitTime 待ち時間
     * @param buffer 読み込みバッファ
     * @param offset 開始位置
     * @param size 最大読み込みサイズ
     * @return 読み込んだサイズ
     * @throws IOException 処理が失敗した場合にスローされます。
     */
    public static int read(I2CDevice i2cDevice, long waitTime, byte[] buffer, int offset, int size) throws IOException {

        long timeLimit = System.currentTimeMillis() + waitTime;
        int len = 0;
        IOException ex = null;
        while (System.currentTimeMillis() <= timeLimit) {
            try {
                if ((len = i2cDevice.read(buffer, offset, size)) >= size) {
                    return len;
                }
            } catch (IOException e) {
                ex = e;
            }
        }
        if (ex != null) {
            throw ex;
        }
        return len;
    }
}
