/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.pi.device.gpio;

import java.util.Date;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import com.wabit.uecs.pi.db.ComponentValueEntity;
import com.wabit.uecs.pi.db.DatabaseUtils;

/**
 * パルス信号検出センサの実装クラスです。
 *
 * @author WaBit
 */
public class PulseSensor<T extends PulseSensorConfig> extends DigitalSensorBase<T> {

    /** カウント方法：総計値 */
    public static final String COUNT_METHOD_SUM = "SUM";

    /** カウント方法：単位時間差分値 */
    public static final String COUNT_METHOD_DIFF = "DIFF";

    // 内部パルスカウンタ
    private long counter;
    // カウンタ上限値
    private long countMax;
    // 検出対象状態
    private boolean countState;
    // 差分カウント処理フラグ
    private boolean isDiffMethod;

    // 差分カウンタ処理スレッド
    private ScheduledExecutorService diffCountExecutor;

    /**
     * コンストラクタです。
     * @param id コンポーネントID
     * @param config 設定値
     */
    public PulseSensor(String id, T config) {
        super(id, config);
    }

    /**
     * 内部パルスカウンタの数を取得します。
     * @return カウント数
     */
    public long getCountNumber() {
        return this.counter;
    }

    /**
     * 内部パルスカウンタを初期化(0)します。
     */
    public void resetCounter() {
        this.counter = 0L;
    }

    @Override
    protected void onInit() throws Exception {
        super.onInit();
        PulseSensorConfig config = getConfig();

        // 通常はオープンコレクタ、負論理で検出
        countState = config.getBoolean(PulseSensorConfig.KEY_COUNT_STATE, false);
        countMax = config.getLong(PulseSensorConfig.KEY_COUNT_MAX, Long.MAX_VALUE);

        if (COUNT_METHOD_DIFF.equals(config.getString(PulseSensorConfig.KEY_COUNT_METHOD, COUNT_METHOD_SUM))) {
            isDiffMethod = true;
        }

        // 総計カウントでデータが記録されている場合は、カウンタ値を復元する
        if (!isDiffMethod) {
            long lastCountTime = config.getLong(PulseSensorConfig.KEY_LAST_COUNT_TIME, Long.MIN_VALUE);
            if (config.getInt(PulseSensorConfig.KEY_RECORD_INTERVAL, 0) > 0) {
                long lastRecMilli = config.getLong(PulseSensorConfig.KEY_LAST_RECORDED_TIME, Long.MIN_VALUE);
                if (lastRecMilli > lastCountTime) {
                    Date time = new Date(lastRecMilli);
                    List<ComponentValueEntity> values = DatabaseUtils.getComponentValues(getId(), time, null);
                    if (values.size() > 0) {
                        this.counter = values.get(0).getValue().longValue();
                    }
                } else {
                    this.counter = config.getLong(PulseSensorConfig.KEY_LAST_COUNT_NUM, 0L);
                }
            } else {
                this.counter = config.getLong(PulseSensorConfig.KEY_LAST_COUNT_NUM, 0L);
            }
        }

    }

    @Override
    protected void onStart() throws Exception {
        // 差分カウンタの場合は、スレッド処理生成
        if (isDiffMethod) {
            long period = getCcmService().getCcm().getLevel().getCycleTime();
            diffCountExecutor = Executors.newSingleThreadScheduledExecutor();
            diffCountExecutor.scheduleAtFixedRate(new Runnable() {
                private long preCount;

                @Override
                public void run() {
                    // カウンタ循環判定
                    if (preCount > counter) {
                        setValue(countMax - preCount + counter);
                    } else {
                        setValue(counter - preCount);
                    }
                    preCount = counter;
                }

            }, 0, period, TimeUnit.MILLISECONDS);
        } else {

            setValue(counter);
        }
        super.onStart();
    }

    @Override
    protected void onStop() throws Exception {
        // 差分カウンタスレッド停止
        if (diffCountExecutor != null) {
            diffCountExecutor.shutdown();
            diffCountExecutor = null;
        }

        if (!isDiffMethod) {
            DatabaseUtils.saveComponentConfig(getId(),
                    PulseSensorConfig.KEY_LAST_COUNT_TIME, Long.toString(System.currentTimeMillis()));
            DatabaseUtils.saveComponentConfig(getId(),
                    PulseSensorConfig.KEY_LAST_COUNT_NUM, Long.toString(counter));

        }

        super.onStop();
    }

    @Override
    protected void onSensePinStateChanged(boolean state) throws Exception {
        if (countState == state) {
            if (counter >= countMax) {
                counter = 0L;
            }
            counter++;
            if (!isDiffMethod) {
                // 差分値計測でない場合は即時セット
                setValue(counter);
            }
        }
    }

}
