/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.pi.device.gpio;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.pi4j.io.gpio.GpioPinDigitalInput;
import com.pi4j.io.gpio.Pin;
import com.pi4j.io.gpio.PinPullResistance;
import com.pi4j.io.gpio.event.GpioPinDigitalStateChangeEvent;
import com.pi4j.io.gpio.event.GpioPinListenerDigital;
import com.pi4j.io.gpio.event.PinEventType;
import com.wabit.uecs.pi.device.PiSensorBase;
import com.wabit.uecs.pi.webui.WebUIApplication;

/**
 * Raspberry PiのGPIOピンに接続されたセンサの基底実装クラスです。
 * @author WaBit
 *
 * @param <T> 設定値クラス
 */
public abstract class DigitalSensorBase<T extends DigitalSensorConfig> extends PiSensorBase<T> {

    /** ピン抵抗：プルアップ */
    public static final String PIN_PULL_UP = PinPullResistance.PULL_UP.name();
    /** ピン抵抗：プルダウン */
    public static final String PIN_PULL_DOWN = PinPullResistance.PULL_DOWN.name();
    /** ピン抵抗：なし */
    public static final String PIN_PULL_OFF = PinPullResistance.OFF.name();

    // 手動モード選択ピン
    private GpioPinDigitalInput senseInputPin;
    // ロガー
    private Log logger = LogFactory.getLog(getClass());

    /**
     * コンストラクタです。
     * @param id コンポーネントID
     * @param config 設定値
     */
    public DigitalSensorBase(String id, T config) {
        super(id, config);
    }

    /**
     * センサ入力ピンを取得します。
     * @return ピンインスタンス(設定されていない場合はnull)
     */
    public Pin getSenseInputPin() {
        DigitalSensorConfig config = getConfig();
        String val = config.getString(DigitalSensorConfig.KEY_GPIO_PIN_SENSE_INPUT);
        if (val != null && !val.isEmpty()) {
            return ((GpioDeviceBase<?>) getDevice()).getPin(val);
        }
        return null;
    }

    @Override
    protected void onStart() throws Exception {
        DigitalSensorConfig config = getConfig();

        // 本番モードのときだけ実際のGPIO初期化
        Pin sensePin = getSenseInputPin();
        GpioDeviceBase<?> device = (GpioDeviceBase<?>) getDevice();
        if (!WebUIApplication.getNodeInstance().isDevelopmentMode() && sensePin != null) {
            // センサ入力ピンの初期化
            senseInputPin = (GpioPinDigitalInput)((GpioDeviceBase<?>) getDevice()).getProvisionedGpioPinInput(sensePin);
            PinPullResistance registance = PinPullResistance.valueOf(config.getString(
                    DigitalSensorConfig.KEY_PIN_PULL_REGISTSANCE, PIN_PULL_OFF));
            if (senseInputPin == null) {
                senseInputPin = device.provisionDigitalInputPin(sensePin, registance);
            }
            if (senseInputPin == null) {
                throw new Exception("GPIO pin not found : " + sensePin);
            }
            logger.debug("sene_input_pin : " + sensePin);

            // デバウンス時間の設定(高頻度パルス検出の抑制)
            int debounce = config.getInt(DigitalSensorConfig.KEY_PIN_INPUT_DEBOUNCE, 0);
            if (debounce > 0) {
                senseInputPin.setDebounce(debounce);
            }

            device.addListener(new GpioPinListenerDigital() {

                @Override
                public void handleGpioPinDigitalStateChangeEvent(
                        GpioPinDigitalStateChangeEvent event) {
                    if (event.getEventType() == PinEventType.DIGITAL_STATE_CHANGE) {
                        try {
                            onSensePinStateChanged(event.getState().isHigh());
                        } catch (Exception e) {
                            logger.error("event process error.", e);
                            DigitalSensorBase.this.notifyException(e);
                        }
                    }

                }
            }, senseInputPin);
        }
    }

    @Override
    protected void onStop() throws Exception {
        // センサ入力ピンの解放
        if (senseInputPin != null) {
            ((GpioDeviceBase<?>) getDevice()).unprovisionPin(senseInputPin);
            senseInputPin = null;
        }
    }

    /**
     * センサ入力ピンの状態が変化したときに起動されるハンドラメソッドです。
     * @param state HIGHに変化した場合はtrue, LOWの場合はfalse
     * @throws Exception 処理に失敗するとスローされます。
     */
    protected abstract void onSensePinStateChanged(boolean state) throws Exception;

}
