/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.pi.device;

import java.sql.SQLException;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.Callable;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.wabit.uecs.device.ComponentConfig;
import com.wabit.uecs.device.IComponent;
import com.wabit.uecs.device.IComponentListener;
import com.wabit.uecs.device.sensor.SensorComponent;
import com.wabit.uecs.pi.AppConstants;
import com.wabit.uecs.pi.UecsPiLogger;
import com.wabit.uecs.pi.db.DatabaseManager;
import com.wabit.uecs.pi.db.DatabaseUtils;
import com.wabit.uecs.pi.util.MessageCode;

/**
 * センサー共通の基底実装クラスです。
 *
 * @author WaBit
 */
public abstract class PiSensorBase<T extends PiSensorConfig> extends SensorComponent<T> {

    /** 値検出方式 */
    private String detectMethod;

    /** 検出時間 */
    private long detectInterval;

    /** スケール換式タイプ */
    private String convertType;

    /** スケール変換係数 a */
    private double coeffA;

    /** スケール変換係数 b */
    private double coeffB;

    /** スケール変換係数 c */
    private double coeffC;

    /** サーミスタ計算用：プルアップ抵抗値 */
    private double pullupR;

    /** サーミスタ計算用：基準抵抗値 */
    private double stdR;

    /** サーミスタ計算用：基準温度 */
    private double stdTemp;

    /** サーミスタ計算用：B定数 */
    private double constB;

    /** 電源電圧 */
    private double powerV;

    /** 記録間隔 */
    private long recInterval;

    /** 最終保存時刻 */
    private long lastRecTime;

    /** 下限値 */
    private double limitMin = Double.NEGATIVE_INFINITY;

    /** 上限値 */
    private double limitMax = Double.POSITIVE_INFINITY;

    /** 計算用値バッファ */
    private List<TsValue> values = new LinkedList<TsValue>();

    // ロガー
    private Log logger = LogFactory.getLog(getClass());

    /** 時系列データ */
    private static class TsValue {
        long time;
        double value;
    }

    /**
     * コンストラクタです。
     * @param id コンポーネントID
     * @param config 設定値
     */
    public PiSensorBase(String id, T config) {
        super(id, config);
        addListener(new SensorListener());
    }

    @Override
    protected void onInit() throws Exception {
        PiSensorConfig config = getConfig();
        detectMethod = config.getString(PiSensorConfig.KEY_DETECTION_METHOD, PiSensorConfig.DETECTION_MOMENT);
        detectInterval = config.getLong(PiSensorConfig.KEY_DETECTION_INTERVAL, 0L) * 1000L;
        convertType = config.getString(PiSensorConfig.KEY_CONV_TYPE, PiSensorConfig.CONV_TYPE_QUADRATIC);
        coeffA = config.getDouble(PiSensorConfig.KEY_COEFF_2, 0);
        coeffB = config.getDouble(PiSensorConfig.KEY_COEFF_1, 1);
        coeffC = config.getDouble(PiSensorConfig.KEY_COEFF_0, 0);
        pullupR = config.getDouble(PiSensorConfig.KEY_PULLUP_R, 0);
        stdR = config.getDouble(PiSensorConfig.KEY_STD_R, 0);
        stdTemp = config.getDouble(PiSensorConfig.KEY_STD_TEMP, 0);
        constB = config.getDouble(PiSensorConfig.KEY_CONST_B, 0);
        powerV = config.getDouble(PiSensorConfig.KEY_POWER_V, 0);
        limitMin = config.getDouble(PiSensorConfig.KEY_LIMIT_MIN, Double.NEGATIVE_INFINITY);
        limitMax = config.getDouble(PiSensorConfig.KEY_LIMIT_MAX, Double.POSITIVE_INFINITY);

    }

    /**
     * 変換式や上限、下限制約に基づいた値を設定します。
     */
    @Override
    public synchronized void setValue(Number num) {
        if (num == null) {
            super.setValue(num);
            return;
        }

        double value = num.doubleValue();

        // スケール変換
        if (convertType.equals(PiSensorConfig.CONV_TYPE_QUADRATIC)) {
            // 変換式 y = ax^2 + bx + c で変換
            value = (coeffA * value * value) + (coeffB * value) + coeffC;
        } else if (convertType.equals(PiSensorConfig.CONV_TYPE_THERMISTOR)) {
            // サーミスタ式で変換
            value = calcTemperature(value, pullupR, stdR, stdTemp, constB, powerV);
        }

        // 下限値、上限値制限
        if (value < limitMin) {
            value = limitMin;
        } else if (value > limitMax) {
            value = limitMax;
        }

        if (!detectMethod.equals(PiSensorConfig.DETECTION_MOMENT)) {
            TsValue curVal = new TsValue();
            curVal.time = System.currentTimeMillis();
            curVal.value = value;
            // 古いデータをバッファから削除
            for (TsValue tv : values.toArray(new TsValue[0])) {
                if (tv.time < curVal.time - detectInterval) {
                    values.remove(tv);
                }
            }
            values.add(curVal);

            if (detectMethod.equals(PiSensorConfig.DETECTION_SMA)) {
                // 単純移動平均計算
                value = 0;
                for (TsValue tv : values) {
                    value += tv.value;
                }
                value /= values.size();
            } else if (detectMethod.equals(PiSensorConfig.DETECTION_WMA)) {
                // 加重移動平均計算
                value = 0;
                int div = 0;
                int n = 0;
                for (TsValue tv : values) {
                    value += tv.value * ++n;
                    div += n;
                }
                value /= div;
            }
        }

        super.setValue(value);
    }

    /*
     * サーミスタ分圧抵抗回路の電圧から温度を計算します。
     * @param vth 計測電圧(V)
     * @param r プルアップ抵抗値(Ω)
     * @param r0 基準抵抗(Ω)
     * @param t0 基準温度(℃)
     * @param b B定数
     * @param vp 電源電圧(V)
     * @return 温度(℃)
     */
    private static double calcTemperature(double vth, double r, double r0, double t0, double b, double vp) {
        if (vth > 0 && vp - vth > 0 && r > 0 && r0 > 0 && t0 > -273.15 && b > 0 && vp > 0) {
            double rth = r * vth / (vp - vth);
            double tth = 1 / (1 / b * Math.log(rth / r0) + 1 / (t0 * 1 + 273.15));
            return tth - 273.15;
        } else {
            throw new IllegalArgumentException();
        }
    }

    /*
     * 内部動作リスナー（ログ記録用）
     *
     */
    private class SensorListener implements IComponentListener {

        @Override
        public void componentStarted(IComponent<?> component) {
            if (logger.isDebugEnabled()) {
                logger.debug("ID=" + component.getId());
            }
            recInterval = component.getConfig().getLong(PiSensorConfig.KEY_RECORD_INTERVAL, -1L) * 1000L;
            lastRecTime = component.getConfig().getLong(PiSensorConfig.KEY_LAST_RECORDED_TIME, 0);
        }

        @Override
        public void componentStopped(IComponent<?> component) {
            if (logger.isDebugEnabled()) {
                logger.debug("ID=" + component.getId());
            }
        }

        @Override
        public void componentValueUpdated(final IComponent<?> component) {
            final String compoId = component.getId();
            final Number value = component.getValue();
            if (logger.isDebugEnabled()) {
                logger.debug("ID=" + compoId + ", value=" + value);
            }
            // 保存間隔が設定されていない場合や初期化が完了していない場合は処理を中断
            if (recInterval < 0 || value == null || lastRecTime == Long.MIN_VALUE) {
                return;
            }
            final long time = System.currentTimeMillis();
            if (time - lastRecTime >= recInterval) {
                try {
                    DatabaseManager.callInTransaction( new Callable<Void>() {
                        public Void call() throws Exception {
                            DatabaseUtils.saveComponentData(compoId, time, value);
                            DatabaseUtils.saveComponentConfig(compoId, PiSensorConfig.KEY_LAST_RECORDED_TIME,
                                    Long.toString(time));
                            lastRecTime = time;
                            return null;
                        }
                    }
                        );
                } catch (SQLException e) {
                    logger.error("Data save failed. ID=" + compoId, e);
                    UecsPiLogger.log(UecsPiLogger.CATEGORY_DEVICE, MessageCode.SENSOR_ERROR,
                            new Object[]{component.getConfig().getString(ComponentConfig.KEY_COMPONENT_NAME), e});
                    component.getDevice().getNode().onStatus(AppConstants.STATUS_ARERT_DEVICE_ERROR);
                }
            }
        }

        @Override
        public void handleComponentException(IComponent<?> component, Exception e) {
            logger.error("ID=" + component.getId(), e);
            UecsPiLogger.log(UecsPiLogger.CATEGORY_DEVICE, MessageCode.SENSOR_ERROR,
                    new Object[]{component.getConfig().getString(ComponentConfig.KEY_COMPONENT_NAME), e});
            component.getDevice().getNode().onStatus(AppConstants.STATUS_ARERT_DEVICE_ERROR);
        }

    }
}
