/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.pi.device;

import java.sql.SQLException;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.wabit.uecs.device.ComponentConfig;
import com.wabit.uecs.device.IComponent;
import com.wabit.uecs.device.IComponentListener;
import com.wabit.uecs.device.actuator.ActuatorComponent;
import com.wabit.uecs.pi.UecsPiLogger;
import com.wabit.uecs.pi.db.DatabaseUtils;
import com.wabit.uecs.pi.util.MessageCode;

/**
 * アクチュエータ共通の基底実装クラスです。
 *
 * @author WaBit
 */
public abstract class PiActuatorBase<T extends PiActuatorConfig> extends ActuatorComponent<T> {

    // ロガー
    private Log logger = LogFactory.getLog(getClass());

    /** 記録間隔 */
    private long recInterval;

    private static ScheduledExecutorService recScheduler;

    /**
     * コンストラクタです。
     * @param id コンポーネントID
     * @param config 初期設定
     */
    public PiActuatorBase(String id, T config) {
        super(id, config);
        addListener(new ActuatorListener());
    }

    protected void onInit() throws Exception {
        recInterval = getConfig().getLong(PiActuatorConfig.KEY_RECORD_INTERVAL, 0)  * 1000L;
    }

    /*
     * (非 Javadoc)
     * @see com.wabit.uecs.device.AbstractComponent#onStart()
     */
    @Override
    protected void onStart() throws Exception {

        // 記録間隔が設定されていたら定期保存処理登録
        if (recInterval > 0) {
            // 内部データ記録スレッドを起動
            if (recScheduler == null) {
                recScheduler = Executors.newSingleThreadScheduledExecutor();
            }
            long delay = recInterval - (System.currentTimeMillis() % recInterval);
            recScheduler.scheduleAtFixedRate(new RecordTask(this),
                    delay, recInterval, TimeUnit.MILLISECONDS);
        }
    }

    /*
     * (非 Javadoc)
     * @see com.wabit.uecs.device.AbstractComponent#onStop()
     */
    @Override
    protected void onStop() throws Exception {
        // 内部データ記録スレッドを停止します。
        if (recScheduler != null && !recScheduler.isShutdown()) {
            recScheduler.shutdown();
            recScheduler.awaitTermination(5000, TimeUnit.MILLISECONDS);
            recScheduler = null;
        }
    }

    /**
     * アクチュエータの状態が変更されると内部DBに記録されます。
     */
    @Override
    public void setValue(Number value) {
        super.setValue(value);
    }

    /*
     * 内部動作リスナー（ログ記録用）
     *
     */
    private class ActuatorListener implements IComponentListener {

        private Number preValue;
        private boolean logEnabled;

        @Override
        public void componentStarted(IComponent<?> component) {
            preValue = component.getValue();
            if (logger.isDebugEnabled()) {
                logger.debug("ID=" + component.getId());
            }
            logEnabled = getConfig().getBoolean(PiActuatorConfig.KEY_LOG_ENABLED, false);
        }

        @Override
        public void componentStopped(IComponent<?> component) {
            if (logger.isDebugEnabled()) {
                logger.debug("ID=" + component.getId());
            }
        }

        @Override
        public void componentValueUpdated(IComponent<?> component) {
            String compoId = component.getId();
            Number value = component.getValue();
            if (logger.isDebugEnabled()) {
                logger.debug("ID=" + compoId  + ", value=" + value);
            }
            if (recInterval >= 0 && value != null
                    && (preValue == null || !preValue.equals(value))) {
                    // 値が変化したときに保存する
                try {
                    long time = System.currentTimeMillis();
                    DatabaseUtils.saveComponentData(compoId, time, value);
                } catch (SQLException e) {
                    logger.error("Data save failed. ID=" + compoId, e);
                }
                if (logEnabled) {
                    String category = UecsPiLogger.CATEGORY_DEVICE;
                    String code = MessageCode.COMPONENT_CHANGED;
                    String name = component.getConfig().getString(ComponentConfig.KEY_COMPONENT_NAME);
                    String mode = getActionMode().toString();
                    if (component instanceof IPositioner) {
                        UecsPiLogger.log(category, code, name, mode, value + "%");
                    } else if (component instanceof ISwitcher) {
                        if (((ISwitcher)component).isOn()) {
                            UecsPiLogger.log(category, code, name, mode, "ON");
                        } else {
                            UecsPiLogger.log(category, code, name, mode, "OFF");
                        }
                    } else {
                        UecsPiLogger.log(category, code, name, mode, value);
                    }
                }
            }
            preValue = value;
        }

        @Override
        public void handleComponentException(IComponent<?> component, Exception e) {
            logger.error("ID=" + component.getId(), e);
        }
    }

    /*
     * 定期保存タスク
     */
    private static class RecordTask implements Runnable {

        private PiActuatorBase<?> act;

        RecordTask(PiActuatorBase<?> act) {
            this.act = act;
        }

        @Override
        public void run() {
            long current = System.currentTimeMillis();
            Number value = act.getValue();
            if (value != null) {
                try {
                    long time = current - (current % act.recInterval);
                    DatabaseUtils.saveComponentData(act.getId(), time, value);
                } catch (SQLException e) {
                    act.logger.error("Data save failed. ID=" + act.getId(), e);
                }
            }

        }

    }

}
