/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.pi.device.gpio;

import java.util.concurrent.Callable;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.j256.ormlite.dao.Dao;
import com.pi4j.io.gpio.GpioPinAnalogInput;
import com.pi4j.io.gpio.GpioPinAnalogOutput;
import com.pi4j.io.gpio.GpioPinOutput;
import com.pi4j.io.gpio.GpioPinPwmOutput;
import com.pi4j.io.gpio.Pin;
import com.wabit.uecs.ActionMode;
import com.wabit.uecs.pi.db.ComponentConfigEntity;
import com.wabit.uecs.pi.db.ComponentEntity;
import com.wabit.uecs.pi.db.DatabaseManager;
import com.wabit.uecs.pi.device.IPositioner;
import com.wabit.uecs.pi.webui.WebUIApplication;

/**
 * PWM出力を想定した位置制御アクチュエータの実装クラスです。
 * PWM出力のGPIOピンで制御します。
 *
 * @author WaBit
 */
public class AnalogPositioningActuator<T extends AnalogPositioningActuatorConfig> extends GpioActuatorBase<T> implements
        IPositioner {

    private GpioPinOutput gpioPinOutput;
    private GpioPinAnalogInput gpioPinInput;

    private boolean pwmMode;
    private int pwmRange;

    private Log logger = LogFactory.getLog(getClass());
    /**
     * コンストラクタです。
     * @param id コンポーネントID
     * @param config 設定値
     */
    public AnalogPositioningActuator(String id, T config) {
        super(id, config);
    }

    /**
     * 制御出力ピンを取得します。
     * @return ピンインスタンス
     */
    public Pin getOutputPin() {
        String pinName = getConfig().getString(AnalogPositioningActuatorConfig.KEY_GPIO_PIN_OUT);
        if (pinName != null && !pinName.isEmpty()) {
            return ((GpioDeviceBase<?>) getDevice()).getPin(pinName);
        }
        return null;
    }

    /**
     * 制御出力ピンを取得します。
     * @return ピンインスタンス
     */
    public Pin getInputPin() {
        String pinName = getConfig().getString(AnalogPositioningActuatorConfig.KEY_GPIO_PIN_IN);
        if (pinName != null && !pinName.isEmpty()) {
            return ((GpioDeviceBase<?>) getDevice()).getPin(pinName);
        }
        return null;
    }

    /**
     * 内部保持しているGPIO入力ピンを取得します。
     * @return ピンインスタンス
     */
    GpioPinAnalogInput getGpioInputPin() {
        return this.gpioPinInput;
    }

    /**
     * GPIOピンを初期化し、rcM/rcA受信動作アクションを登録します。
     */
    @Override
    protected void onInit() throws Exception {
        super.onInit();

        if (AnalogPositioningActuatorConfig.VAL_PIN_MODE_PWM.equals(
                getConfig().getString(AnalogPositioningActuatorConfig.KEY_GPIO_PIN_OUT_MODE))) {
            pwmMode = true;
            pwmRange = getConfig().getInt(AnalogPositioningActuatorConfig.KEY_ANALOG_RANGE_OUT, 100);
        } else {
            pwmMode = false;
        }

        // インターロック動作
        setAction(ActionMode.Interlock, new AnalogPositioningInterlockAction());

        // スタンドアローンモード動作
        FixedPositioningAction stdAction = new FixedPositioningAction();
        stdAction.setPosition(getConfig().getInt(AnalogPositioningActuatorConfig.KEY_DEFAULT_VALUE, 0));
        setAction(ActionMode.Standalone, stdAction);

    }

    /*
     * (非 Javadoc)
     * @see com.wabit.uecs.pi.device.gpio.GpioActuatorBase#onStart()
     */
    @Override
    protected void onStart() throws Exception {
        GpioDeviceBase<?> device = (GpioDeviceBase<?>) getDevice();

        // 本番モードのときだけ実際のGPIO操作
        if (!WebUIApplication.getNodeInstance().isDevelopmentMode()) {
            // 正方向制御出力ピン取得
            Pin pin = getOutputPin();
            if (pin != null && gpioPinOutput == null) {
                if (pwmMode) {
                    logger.debug("PWM mode PIN = " + pin);
                    gpioPinOutput = device.provisionPwmOutputPin(pin);
                    ((GpioPinPwmOutput)gpioPinOutput).setPwmRange(pwmRange);
                } else {
                    logger.debug("ANALOG mode PIN = " + pin);
                    gpioPinOutput = device.provisionAnalogOutputPin(pin);
                }
                if (gpioPinOutput == null) {
                    throw new Exception("GPIO pin not found : " + pin);
                }
            }

            pin = getInputPin();
            if (pin != null && gpioPinInput == null) {
                gpioPinInput = device.provisionAnalogInputPin(pin);
            }
        }

        // アナログ初期値セット
        if (getValue() != null) {
            setPinValue(getValue().intValue());
        }
        super.onStart();
    }

    /*
     * (非 Javadoc)
     * @see com.wabit.uecs.pi.device.gpio.GpioActuatorBase#onStop()
     */
    @Override
    protected void onStop() throws Exception {
        DatabaseManager.callInTransaction(new Callable<Void>() {
            public Void call() throws Exception {
                Dao<ComponentEntity, String> compDao = DatabaseManager.createDao(ComponentEntity.class);
                if (compDao.queryForId(getId()) != null) {
                    Dao<ComponentConfigEntity, String> confDao = DatabaseManager.createDao(ComponentConfigEntity.class);

                    ComponentConfigEntity ent = new ComponentConfigEntity();
                    ent.setComponentId(getId());
                    ent.setId(getId() + "." + AnalogPositioningActuatorConfig.KEY_LAST_VALUE);
                    ent.setKey(AnalogPositioningActuatorConfig.KEY_LAST_VALUE);
                    ent.setValue(Integer.toString(getPosition()));
                    confDao.createOrUpdate(ent);
                }
                return null;
            }
        });

        GpioDeviceBase<?> device = (GpioDeviceBase<?>) getDevice();

        // 本番モードのときだけ実際のGPIO操作
        if (gpioPinOutput != null) {
            device.unprovisionPin(gpioPinOutput);
            gpioPinOutput = null;
        }

        if (gpioPinInput != null) {
            device.unprovisionPin(gpioPinInput);
            gpioPinInput = null;
        }
        super.onStop();
    }

    /*
     * (非 Javadoc)
     * @see com.wabit.uecs.pi.device.IPositioner#getPosition()
     */
    @Override
    public int getPosition() {
        if (getValue() == null) {
            return 0;
        } else {
            return getValue().intValue();
        }
    }

    /*
     * (非 Javadoc)
     * @see com.wabit.uecs.pi.device.IPositioner#setPosition(int)
     */
    @Override
    public void setPosition(int pos) {
        // 範囲外例外
        if (pos < 0 || pos > 100) {
            throw new IllegalArgumentException("out of range (0-100) : " + pos);
        }

        int curPos = getPosition();
        if (curPos != pos) {
            setPinValue(pos);
            setValue(pos);
        }
    }

    // ピン出力内部処理
    private void setPinValue(int pos) {
        if (gpioPinOutput != null) {
            int aval = (int)Math.round(pwmRange * (pos / 100.0));
            logger.debug("analog value =" + aval);
            if (pwmMode) {
                ((GpioPinPwmOutput)gpioPinOutput).setPwm(aval);
            } else {
                ((GpioPinAnalogOutput)gpioPinOutput).setValue(aval);
            }
        }
    }

}
