/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.pi.device;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.wabit.uecs.IUecsNode;
import com.wabit.uecs.device.AbstractDevice;
import com.wabit.uecs.device.DeviceConfig;
import com.wabit.uecs.device.DeviceException;
import com.wabit.uecs.device.IComponent;
import com.wabit.uecs.device.IDevice;
import com.wabit.uecs.device.IDeviceListener;
import com.wabit.uecs.pi.AppConstants;
import com.wabit.uecs.pi.UecsPiLogger;
import com.wabit.uecs.pi.db.DatabaseUtils;
import com.wabit.uecs.pi.util.MessageCode;

/**
 * UECS-Piノードで動作するデバイスの基底実装クラスです。
 *
 * @author WaBit
 */
public abstract class PiDeviceBase<T extends PiDeviceConfig> extends AbstractDevice<T>  {

    // ロガー
    private Log logger = LogFactory.getLog(getClass());

    private boolean enabled = true;

    /**
     * コンストラクタです。
     * @param deviceId デバイスID
     * @param config 設定値
     */
    protected PiDeviceBase(String deviceId, T config) {
        super(deviceId, config);
        addListener(new DeviceListener());
    }


    /**
     * デバイスが有効状態であるかを返します。
     * @return trueで有効
     */
    public boolean isEnabled() {
        return enabled;
    }

    /**
     * デバイスが有効状態であるかを設定します。
     * @param enabled フラグ
     */
    public void setEnabled(boolean enabled) {
        this.enabled = enabled;
    }

    @Override
    public void init(IUecsNode<?> node) throws DeviceException {
        if (isEnabled()) {
            super.init(node);
        }
    }

    @Override
    public void start() throws DeviceException {
        if (isEnabled()) {
            super.start();
        }
    }

    @Override
    public void stop() throws DeviceException {
        if (isEnabled()) {
            super.stop();
        }
    }

    /**
     * DBからコンポーネント情報を読み込んで登録します。
     */
    @Override
    protected void onInit() throws Exception {
        for (IComponent<?> compo : DatabaseUtils.loadComponents(getId())) {
            addComponent(compo);
        }
    }

    @Override
    protected void onStart() throws Exception {
        if (logger.isDebugEnabled()) {
            logger.debug("ID=" + getId());
        }
    }

    @Override
    protected void onStop() throws Exception {
        if (logger.isDebugEnabled()) {
            logger.debug("ID=" + getId());
        }
    }


    /*
     * 内部デバイス動作リスナー(ログ記録用)
     */
    private class DeviceListener implements IDeviceListener {
        @Override
        public void deviceStarted(IDevice<?> device) {
            if (logger.isDebugEnabled()) {
                logger.debug("ID=" + device.getId());
            }
        }

        @Override
        public void deviceStopped(IDevice<?> device) {
            if (logger.isDebugEnabled()) {
                logger.debug("ID=" + device.getId());
            }
        }

        @Override
        public void handleDeviceException(IDevice<?> device, Exception e) {
            logger.error("ID=" + device.getId(), e);
            UecsPiLogger.log(UecsPiLogger.CATEGORY_DEVICE,
                    MessageCode.DEVICE_ERROR, device.getConfig().getString(DeviceConfig.KEY_DEVICE_NAME), e);
            getNode().onStatus(AppConstants.STATUS_ARERT_DEVICE_ERROR);
        }
    }


}
