// m'sスケール用 計測制御IFプログラム for Arduino Mega 2560
#include <Arduino.h>
#include <IcsSoftSerialClass.h>

#define DATASET_NUM 10
#define BUFFER_SIZE 23
 
// KRSサーボICS通信設定
#define CENTER_POS 7500
#define MIN_POS 3500
#define MAX_POS 11500
#define ANGLE_RATE 29.63
const byte EN_PIN_A = 11;
const byte S_TX_PIN_A = 12;
const byte S_RX_PIN_A = 13;
const byte EN_PIN_B = 2;
const byte S_TX_PIN_B = 3;
const byte S_RX_PIN_B = 4;
const long BAUDRATE = 115200;
const int TIMEOUT = 200;   //softSerialは通信失敗する可能性があるため短めに
//インスタンス＋ENピン(2番ピン)およびUARTの設定、softSerial版
IcsSoftSerialClass krsA(S_RX_PIN_A,S_TX_PIN_A,EN_PIN_A,BAUDRATE,TIMEOUT);
IcsSoftSerialClass krsB(S_RX_PIN_B,S_TX_PIN_B,EN_PIN_B,BAUDRATE,TIMEOUT);
IcsSoftSerialClass icsIF[] = {krsA, krsB};

// フットスイッチ入力
const byte FOOT_SW_PIN = 53;

// HX711ロードセルADC通信設定
int hx711Pins[DATASET_NUM][2] = {
    {22, 23},
    {24, 25},
    {26, 27},
    {28, 29},
    {30, 31},
    {32, 33},
    {34, 35},
    {36, 37},
    {38, 39},
    {40, 41},
    };
/*
int hx711Pins[DATASET_NUM][2] = {
  {22, 23},
  {22, 23},
  {22, 23},
  {22, 23},
  {22, 23},
  {22, 23},
  {22, 23},
  {22, 23},
  {22, 23},
  {22, 23},
  };
*/
long dataset[DATASET_NUM];

// シリアル通信バッファ
byte inputBuffer[BUFFER_SIZE];
int readIndex;

void setup() {
  // USBシリアル通信を初期化
  readIndex = 0;
  Serial.begin(115200);

  // フットスイッチ入力を初期化
  pinMode(FOOT_SW_PIN, INPUT_PULLUP);

  // ロードセル通信を初期化
  for (int i=0; i<DATASET_NUM; i++) {
    AE_HX711_Init(i);
    AE_HX711_Reset(i);
  }
  // サーボモータ通信を初期化
  krsA.begin();
  krsB.begin();
}

void loop() {
  if (Serial.available() <= 0) {
    return;
  }
  int inByte = Serial.read();
  switch (inByte) {
    case '\n' : {
      execCommand();
      readIndex = 0;
      break;
    }
    case ':' : {
      readIndex = 0;
    }
    default : {
      inputBuffer[readIndex++] = inByte;
    }
  }
  if (readIndex >= BUFFER_SIZE) {
    readIndex = 0;
  }
}

void execCommand() {
  // 不正コマンド
  if (inputBuffer[0] == ':') {
    switch (inputBuffer[1]) {
      case 'M' : {
        int mode = parseIntParam(3, 4);
        if (mode == 0 && digitalRead(FOOT_SW_PIN) == HIGH) {
          // フットスイッチOFFのときは一時停止(PAUSE)
          Serial.println(":P");
        } else {
          measureLoadCell(parseIntParam(8, 4), parseIntParam(13, 4));
        }
        break;
      }
      case 'A' : {
        actuateServo(parseIntParam(3, 1), parseIntParam(5, 4)
                , parseIntParam(10, 3), parseIntParam(14, 3)
                , parseIntParam(18, 5));
        break;
      }
      case 'R' : {
        rotateServo(parseIntParam(3, 1), parseIntParam(5, 4)
                , parseIntParam(10, 3), parseIntParam(14, 5));
        break;
      }
      default : {
        Serial.println(":E");
      }
    }
  } else {
    Serial.println(":E");
  }
}

int parseIntParam(int stIdx, int len) {
  // 数字文字列を抽出する
  char dec_substr[len + 1]; // +1 は '\0' を含めるため
  strncpy(dec_substr, inputBuffer + stIdx, len);
  dec_substr[len] = '\0'; // 文字列の終端を明示する
  char *endptr;
  return strtol(dec_substr, &endptr, DEC);
}

void measureLoadCell(int target, int aveNum) {
  //Serial.print("read load cell. average=");
  //Serial.println(aveNum);
  for (int i = 0; i < DATASET_NUM; i++) {
    dataset[i] = 0;
  }
  for (int i = 0; i < aveNum; i++) {
    for (int j = 0; j < DATASET_NUM; j++) {
      if (((1 << j) & target) != 0) {
        dataset[j] += AE_HX711_Read(j);
      }
    }
  }
  Serial.print(":M");
  for (int i = 0; i < DATASET_NUM; i++) {
    Serial.print(',');
    Serial.print(dataset[i] / aveNum, HEX);
  }
  Serial.println();
}

void actuateServo(int index, int target, int startAngle, int endAngle, int interval) {
  /*
  Serial.print("servo index=");
  Serial.print(index);
  Serial.print(", target=");
  Serial.print(target, BIN);
  Serial.print(", angle=");
  Serial.println(angle);
  */
  setPos(index, target, startAngle);
  delay(interval);
  setPos(index, target, endAngle);
  Serial.println(":A");
}

void setPos(int index, int target, int angle) {
  int pos = (int)(CENTER_POS + (angle * ANGLE_RATE));
  if (pos >= MIN_POS && pos <= MAX_POS) {
    for (int i = 0; i < DATASET_NUM; i++) {
      if (((1 << i) & target) != 0) {
        icsIF[index].setPos(i+1, pos);
      }
    }
  }
}

void rotateServo(int index, int target, int angle, int interval) {
  int pos = (int)(CENTER_POS + (angle * ANGLE_RATE));
  if (pos >= MIN_POS && pos <= MAX_POS) {
    for (int i = 0; i < DATASET_NUM; i++) {
      if (((1 << i) & target) != 0) {
        icsIF[index].setPos(i+1, pos);
        delay(interval);
        icsIF[index].setPos(i+1, CENTER_POS);        
      }
    }
  }
  Serial.println(":R");
}

void AE_HX711_Init(int no)
{
  pinMode(hx711Pins[no][0], INPUT);
  pinMode(hx711Pins[no][1], OUTPUT);
}

void AE_HX711_Reset(int no)
{
  digitalWrite(hx711Pins[no][1],1);
  delayMicroseconds(100);
  digitalWrite(hx711Pins[no][1],0);
  delayMicroseconds(100); 
}

long AE_HX711_Read(int no)
{
  long data=0;
  while(digitalRead(hx711Pins[no][0])!=0);
  delayMicroseconds(10);
  for(int i=0;i<24;i++)
  {
    digitalWrite(hx711Pins[no][1],1);
    delayMicroseconds(5);
    digitalWrite(hx711Pins[no][1],0);
    delayMicroseconds(5);
    data = (data<<1)|(digitalRead(hx711Pins[no][0]));
  }
  //Serial.println(data,HEX);   
  digitalWrite(hx711Pins[no][1],1);
  delayMicroseconds(10);
  digitalWrite(hx711Pins[no][1],0);
  delayMicroseconds(10);
  return data^0x800000; 
}