/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.pi.device.serial;

import gnu.io.CommPortIdentifier;
import gnu.io.SerialPort;
import gnu.io.SerialPortEvent;
import gnu.io.SerialPortEventListener;

import java.io.InputStream;
import java.io.OutputStream;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.wabit.uecs.pi.util.BytesUtils;

/**
 * Gnu RX/TXライブラリを用いたシリアル通信クライアントクラスです。
 * @author WaBit
 */
public class SerialPortClientGnuIoImpl extends SerialPortClientBase implements SerialPortEventListener {
    private SerialPort serialPort;
    private InputStream input;
    private OutputStream output;
    private Log logger = LogFactory.getLog(getClass());
    private String port;
    private int speed;

    @Override
    public void init(SerialPortDeviceConfig conf) throws Exception {
        this.port = conf.getString(SerialPortDeviceConfig.KEY_SERIAL_PORT_ID);
        if (port != null) {
            logger.info("port ID=" + port);
        }
        this.speed = conf.getInt(SerialPortDeviceConfig.KEY_SERIAL_PORT_SPEED, 9600);
    }

    @Override
    public void open() throws Exception {
        if (serialPort != null) {
            return;
        }
        CommPortIdentifier portId = CommPortIdentifier.getPortIdentifier(port);
        logger.info("open.");
        serialPort = (SerialPort) portId.open("SerialPortClient", 1500);
        serialPort.setSerialPortParams(speed, // 通信速度[bps]
            SerialPort.DATABITS_8, // データビット数
            SerialPort.STOPBITS_1, // ストップビット
            SerialPort.PARITY_NONE // パリティ
        );
        serialPort.setFlowControlMode(SerialPort.FLOWCONTROL_NONE);
        input = serialPort.getInputStream();
        output = serialPort.getOutputStream();
        serialPort.addEventListener(this);
        serialPort.notifyOnDataAvailable(true);
    }

    @Override
    public void close() throws Exception {
        try {
            if (input != null) {
                input.close();
                input = null;
            }
            if (output != null) {
                output.close();
                output = null;
            }
        } catch (Exception e) {
            logger.warn(e, e);
            throw e;
        } finally {
            if (serialPort != null) {
                serialPort.notifyOnDataAvailable(false);
                serialPort.close();
                serialPort = null;
            }
            // ポートが閉じられるまで非同期で少し時間がかかるので少し待つ
            try {
                Thread.sleep(1000L);
            } catch (InterruptedException e) {

            }
        }
    }


    @Override
    public void sendData(byte[] data) throws Exception {
        if (logger.isDebugEnabled()) {
            logger.debug("write data : "  + BytesUtils.toHexString(data));
        }
        output.write(data);
        output.flush();
    }

    @Override
    public void serialEvent(SerialPortEvent event) {
        if (SerialPortEvent.DATA_AVAILABLE != event.getEventType()) {
            return;
        }
        try {
            int size;
            byte[] data = null;
            while ((size = input.available()) > 0) {
                data = new byte[size];
                input.read(data, 0, size);
                if (logger.isDebugEnabled()) {
                    logger.debug("read size : "  + size + "bytes");
                    logger.debug("read data : "  + BytesUtils.toHexString(data));
                }
                fireDataAvailableEvent(data);
            }
        } catch (Exception e) {
            logger.error(e);
        }
    }

}
