/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.pi.device.i2c;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import com.pi4j.io.i2c.I2CBus;
import com.pi4j.io.i2c.I2CDevice;
import com.pi4j.io.i2c.I2CFactory;
import com.wabit.uecs.pi.device.PiDeviceBase;

/**
 * Raspberry PiのI2Cピンに接続されたデバイスの基底実装クラスです。
 *
 * @author WaBit
 */
public class I2cDeviceBase<T extends I2cDeviceConfig> extends PiDeviceBase<T> implements I2cReader {

    private static I2CBus I2C_BUS;
    private I2CDevice i2cDevice;
    private static ScheduledExecutorService executorService;

    /**
     * センサー値取得タスク
     *
     */
    private class SenseTask implements Runnable {
        private I2cReader reader;

        SenseTask(I2cReader reader) {
            this.reader = reader;
        }

        @Override
        public void run() {
            try {
                reader.onReadProcess(i2cDevice);
            } catch (Exception e) {
                reader.onReadError(e);
                I2cDeviceBase.this.notifyException(e);
            }
        }

    }

    /**
     * コンストラクタです。
     * @param deviceId デバイスID
     * @param config 設定値
     */
    public I2cDeviceBase(String deviceId, T config) {
        super(deviceId, config);
    }


    /**
     * 内部でI2Cバスを初期化して接続します。
     */
    @Override
    protected void onInit() throws Exception {
        super.onInit();

        // I2C通信クライアントを設定値で切り替える
        String clsClass = getConfig().getString(I2cDeviceConfig.KEY_I2C_CLIENT_CLASS);
        if (clsClass == null || PiI2CDeviceMockImpl.class.getName().equals(clsClass)) {
            i2cDevice = new PiI2CDeviceMockImpl();
        } else if (com.pi4j.io.i2c.impl.I2CDeviceImpl.class.getName().equals(clsClass)) {
            if (I2C_BUS == null) {
                I2C_BUS = I2CFactory.getInstance(getConfig().getInt(I2cDeviceConfig.KEY_I2C_BUS));
            }
            i2cDevice = I2C_BUS.getDevice(
                    getConfig().getInt(I2cDeviceConfig.KEY_I2C_ADDRESS));
        } else {
            i2cDevice = (I2CDevice)Class.forName(clsClass).newInstance();
        }
    }

    /**
     * センサー値取得スレッドを起動します。
     */
    @Override
    protected void onStart() throws Exception {
        // バス接続が競合しないように、シングルトンなスレッドでアクセス実行させる
        if (executorService == null) {
            executorService = Executors.newSingleThreadScheduledExecutor();
        }

        int period = getConfig().getInt(I2cDeviceConfig.KEY_SENSE_INTERVAL, 0);
        if (period > 0) {
            addI2cReaderTask(this, period, TimeUnit.SECONDS);
        }
    }

    @Override
    protected void onStop() throws Exception {
        if (executorService != null) {
            executorService.shutdownNow();
            executorService = null;
        }
    }

    /**
     * 読み込みタスクを登録します。
     * @param reader
     * @param period
     * @param unit
     */
    public void addI2cReaderTask(I2cReader reader, long period, TimeUnit unit) {
        executorService.scheduleAtFixedRate(new SenseTask(reader), 0L, period, unit);
    }

    /**
     * PI4JのI2C接続クラスを取得します。
     * @return I2C接続クラス
     */
    protected I2CDevice getI2CDevice() {
        return i2cDevice;
    }


    /**
     * デフォルトは空実装です。
     */
    @Override
    public void onReadProcess(I2CDevice i2cDevice) throws Exception {

    }

    /**
     * デフォルトは空実装です。
     */
    @Override
    public void onReadError(Exception e) {

    }


}
