/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.pi.device.gpio;

import java.util.concurrent.Callable;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.j256.ormlite.dao.Dao;
import com.pi4j.io.gpio.GpioPinDigitalInput;
import com.pi4j.io.gpio.GpioPinDigitalOutput;
import com.pi4j.io.gpio.Pin;
import com.pi4j.io.gpio.PinPullResistance;
import com.pi4j.io.gpio.PinState;
import com.wabit.uecs.ActionMode;
import com.wabit.uecs.pi.db.ComponentConfigEntity;
import com.wabit.uecs.pi.db.ComponentEntity;
import com.wabit.uecs.pi.db.DatabaseManager;
import com.wabit.uecs.pi.device.IPositioner;
import com.wabit.uecs.pi.webui.WebUIApplication;

/**
 * ２個のリレー接続を想定した位置制御アクチュエータの実装クラスです。
 * 動作ON/OFF、正/逆を２つのGPIOピンで制御します。
 *
 * @author WaBit
 */
public class DigitalPositioningActuator<T extends DigitalPositioningActuatorConfig> extends GpioActuatorBase<T> implements IPositioner {
    /** 状態定数：正方向動作中 */
    public static final int UP = 1;
    /** 状態定数：逆方向動作中 */
    public static final int DOWN = -1;
    /** 状態定数：停止 */
    public static final int STOP = 0;


    private GpioPinDigitalOutput gpioPinUpOutput;
    private GpioPinDigitalOutput gpioPinDownOutput;
    private GpioPinDigitalInput gpioPinUpInput;
    private GpioPinDigitalInput gpioPinDownInput;

    private Log logger = LogFactory.getLog(getClass());

    // 最大位置までの動作時間
    private long maxTimeMillis;
    // 実行スレッドサービス
    private ScheduledExecutorService executor;
    // 現在動作状態
    private int actionStatus;
    // 動作開始時刻
    private long actionStartTime;
    // 動作開始位置
    private double actionStartPosition;
    // 現在位置
    private double currentPosition;
    // 動作時間
    private long actionInterval;
    // 位置ギャップ補正フラグ
    private boolean correctionEnabled;

    /**
     * コンストラクタです。
     * @param id コンポーネントID
     * @param config 設定値
     */
    public DigitalPositioningActuator(String id, T config) {
        super(id, config);
    }


    /**
     * 現在の動作状態を取得します。
     * @return 動作状態定数
     */
    int getActionStatus() {
        return actionStatus;
    }

    /**
     * 正方向制御出力ピンを取得します。
     * @return ピンインスタンス
     */
    public Pin getUpOutputPin() {
        String pinName = getConfig().getString(DigitalPositioningActuatorConfig.KEY_GPIO_PIN_UP_OUT);
        if (pinName != null && !pinName.isEmpty()) {
            return ((GpioDeviceBase<?>) getDevice()).getPin(pinName);
        }
        return null;
    }
    /**
     * 正方向制御入力ピンを取得します。
     * @return ピンインスタンス
     */
    public Pin getUpInputPin() {
        String pinName = getConfig().getString(DigitalPositioningActuatorConfig.KEY_GPIO_PIN_UP_IN);
        if (pinName != null && !pinName.isEmpty()) {
            return ((GpioDeviceBase<?>) getDevice()).getPin(pinName);
        }
        return null;
    }

    /**
     * 負方向制御出力ピンを取得します。
     * @return ピンインスタンス
     */
    public Pin getDownOutputPin() {
        String pinName = getConfig().getString(DigitalPositioningActuatorConfig.KEY_GPIO_PIN_DOWN_OUT);
        if (pinName != null && !pinName.isEmpty()) {
            return ((GpioDeviceBase<?>) getDevice()).getPin(pinName);
        }
        return null;
    }

    /**
     * 負方向制御入力ピンを取得します。
     * @return ピンインスタンス
     */
    public Pin getDownInputPin() {
        String pinName = getConfig().getString(DigitalPositioningActuatorConfig.KEY_GPIO_PIN_DOWN_IN);
        if (pinName != null && !pinName.isEmpty()) {
            return ((GpioDeviceBase<?>) getDevice()).getPin(pinName);
        }
        return null;
    }

    /**
     * GPIOピンを初期化し、rcM/rcA受信動作アクションを登録します。
     */
    @Override
    protected void onInit() throws Exception {
        super.onInit();

        // 起動前の位置を復元(1%未満は切り捨てられる)
        if (getValue() != null) {
            currentPosition = getValue().doubleValue();
            logger.debug("init position : " + currentPosition);
        }

        T config = getConfig();
        correctionEnabled = config.getBoolean(DigitalPositioningActuatorConfig.KEY_GAP_CORRECTION, false);
        maxTimeMillis = config.getLong(DigitalPositioningActuatorConfig.KEY_MAX_POSITION_TIME, 0L) * 1000L;
        actionStatus = STOP;

        // インターロック動作
        setAction(ActionMode.Interlock, new DigitalPositioningInterlockAction());

        // スタンドアローンモード動作
        FixedPositioningAction stdAction = new FixedPositioningAction();
        stdAction.setPosition(getConfig().getInt(DigitalPositioningActuatorConfig.KEY_DEFAULT_VALUE, 0));
        setAction(ActionMode.Standalone, stdAction);

    }

    /**
     * 設定値から内部動作情報を初期化し、監視スレッドを起動します。
     */
    @Override
    protected void onStart() throws Exception {

        GpioDeviceBase<?> device = (GpioDeviceBase<?>)getDevice();

        // 本番モードのときだけ実際のGPIO操作
        if (!WebUIApplication.getNodeInstance().isDevelopmentMode()) {

            // 正方向制御出力ピン取得
            Pin pin = getUpOutputPin();
            if (pin != null && gpioPinUpOutput == null) {
                gpioPinUpOutput = device.provisionDigitalOutputPin(pin, PinState.getState(isInverse()));
                if (gpioPinUpOutput == null) {
                    throw new Exception("GPIO pin not found : " + gpioPinUpOutput);
                }
            }
            logger.debug("up_pin(OUTPUT) : " + gpioPinUpOutput);

            // 負方向制御出力ピン取得
            pin = getDownOutputPin();
            if (pin != null && gpioPinDownOutput == null) {
                gpioPinDownOutput = device.provisionDigitalOutputPin(pin, PinState.getState(isInverse()));
                if (gpioPinDownOutput == null) {
                    throw new Exception("GPIO pin not found : " + gpioPinDownOutput);
                }
            }
            logger.debug("down_pin(OUTPUT) : " + gpioPinDownOutput);

            // 正方向制御入力ピン取得
            pin = getUpInputPin();
            if (pin != null && gpioPinUpInput == null) {
                gpioPinUpInput = device.provisionDigitalInputPin(pin, PinPullResistance.PULL_UP);
                if (gpioPinUpInput == null) {
                    throw new Exception("GPIO pin not found : " + gpioPinUpInput);
                }
            }
            logger.debug("up_pin(INPUT) : " + gpioPinUpOutput);

            // 負方向制御入力ピン取得
            pin = getDownInputPin();
            if (pin != null && gpioPinDownInput == null) {
                gpioPinDownInput = device.provisionDigitalInputPin(pin, PinPullResistance.PULL_UP);

                if (gpioPinDownInput == null) {
                    throw new Exception("GPIO pin not found : " + gpioPinDownInput);
                }
            }
            logger.debug("down_pin(INPUT) : " + gpioPinDownInput);

        }

        // 100%動作時間が設定されていない=0の場合は何もしない。
        if (maxTimeMillis <= 0) {
            return;
        }

        // 1%刻みの動作間隔に調整する
        long interval = maxTimeMillis / 100;

        // 動作監視するタスクを登録
        executor = Executors.newSingleThreadScheduledExecutor();
        executor.scheduleAtFixedRate(new Runnable() {

            @Override
            public void run() {
                synchronized (DigitalPositioningActuator.this) {
                    if (actionStatus == STOP) {
                        return;
                    }
                    if (actionStartTime == 0) {
                        actionStartTime = System.currentTimeMillis();
                        actionStartPosition = currentPosition;
                        if (actionStatus == UP) {
                            doUpPosition();
                        } else if (actionStatus == DOWN) {
                            doDownPosition();
                        }
                        return;
                    } else {
                        long delay = System.currentTimeMillis() - actionStartTime;
                        if (delay >= actionInterval) {
                            // 動作時間が経過したら停止
                            stopPosition();
                        } else {
                            // 動作中は現在位置を算出
                            detectPosition();
                        }
                    }
                }
            }

        }, 0, interval, TimeUnit.MILLISECONDS);

        super.onStart();
    }

    /**
     * 監視スレッドを停止し、最終位置をDBに保存します。
     */
    @Override
    protected void onStop() throws Exception {
        if (executor != null && !executor.isShutdown()) {
            executor.shutdownNow();
            executor = null;
        }

        // 動作を停止する
        stopPosition();

        DatabaseManager.callInTransaction(new Callable<Void>() {
            public Void call() throws Exception {
                Dao<ComponentEntity, String> compDao = DatabaseManager.createDao(ComponentEntity.class);
                if (compDao.queryForId(getId()) != null) {
                    Dao<ComponentConfigEntity, String> confDao = DatabaseManager.createDao(ComponentConfigEntity.class);

                    ComponentConfigEntity ent = new ComponentConfigEntity();
                    ent.setComponentId(getId());
                    ent.setId(getId() + "." + DigitalPositioningActuatorConfig.KEY_LAST_VALUE);
                    ent.setKey(DigitalPositioningActuatorConfig.KEY_LAST_VALUE);
                    ent.setValue(Integer.toString(getPosition()));
                    confDao.createOrUpdate(ent);
                }
                return null;
            }
        });

        GpioDeviceBase<?> device = (GpioDeviceBase<?>)getDevice();

        // 本番モードのときだけ実際のGPIO操作
        if (gpioPinUpOutput != null) {
            device.unprovisionPin(gpioPinUpOutput);
            gpioPinUpOutput = null;
        }

        if (gpioPinDownOutput != null) {
            device.unprovisionPin(gpioPinDownOutput);
            gpioPinDownOutput = null;
        }

        if (gpioPinUpInput != null) {
            device.unprovisionPin(gpioPinUpInput);
            gpioPinUpInput = null;
        }

        if (gpioPinDownInput != null) {
            device.unprovisionPin(gpioPinDownInput);
            gpioPinDownInput = null;
        }

        super.onStop();
    }

    /*
     * (非 Javadoc)
     * @see com.wabit.uecs.pi.device.IPositioner#getPosition()
     */
    @Override
    public int getPosition() {
        return Math.round((float)currentPosition);
    }

    /*
     * (非 Javadoc)
     * @see com.wabit.uecs.pi.device.IPositioner#setPosition(int)
     */
    @Override
    synchronized public void setPosition(int pos) {
        // 範囲外例外
        if (pos < 0 || pos > 100) {
            throw new IllegalArgumentException("out of range (0-100) : " + pos);
        }

        if (correctionEnabled && pos == 0 && currentPosition != pos) {
            // 0%位置ギャップ補正処理
            // 全動作時間の5%増しで動作させる
            downPosition((long)((double)maxTimeMillis * 1.05));
        } else if (correctionEnabled && pos == 100 && currentPosition !=  pos) {
            // 100%位置ギャップ補正処理
            // 全動作時間の5%増しで動作させる
            upPosition((long)((double)maxTimeMillis * 1.05));
        } else {
            // 通常は位置差分から動作時間を計算
            double diff = pos - currentPosition;
            long executeTime = (long)((double)maxTimeMillis / 100.0 * diff);
            if (executeTime > 0) {
                upPosition(executeTime);
            } else if (executeTime < 0) {
                downPosition(executeTime * -1);
            }
        }
    }

    /**
     * 位置情報を強制リセットします。
     */
    synchronized public void resetPosition() {
        stopPosition();
        setValue(0);
        currentPosition = 0.0;
    }

    /**
     * 位置を正方向に動作させます。
     * @param executeTime 動作時間(msec)
     */
    synchronized public void upPosition(long executeTime) {
        switch (actionStatus) {
        case UP :
            // 既に正方向動作中の場合は、止めずに動作時間を延長する
            long delay = System.currentTimeMillis() - this.actionStartTime;
            if (delay < executeTime) {
                this.actionInterval += executeTime - delay;
            }
            break;
        case DOWN :
            // 既に負方向動作中の場合は、止める
            stopPosition();
        case STOP :
            // 停止状態から起動する
            this.actionInterval = executeTime;
            actionStatus = UP;
        }
    }

    /**
     * 位置を負方向に動作させます。
     * @param executeTime 動作時間(msec)
     */
    synchronized public void downPosition(long executeTime) {
        switch (actionStatus) {
        case DOWN :
            // 既に負方向動作中の場合は、止めずに動作時間を延長する
            long delay = System.currentTimeMillis() - this.actionStartTime;
            if (delay < executeTime) {
                this.actionInterval += executeTime - delay;
            }
            break;
        case UP :
            // 既に負方向動作中の場合は、止める
            stopPosition();
        case STOP :
            // 停止状態から起動する
            this.actionInterval = executeTime;
            actionStatus = DOWN;
        }
    }

    /**
     * 動作停止させます。
     */
    synchronized public  void stopPosition() {
        // 既に停止中の場合は何もしない
        if (actionStatus == STOP) {
            return;
        }
        doStopPosition();
        detectPosition();
        actionStatus = STOP;
        actionStartTime = 0;
        actionInterval = 0;
    }

    // 現在位置を算出して状態値に反映する
    private void detectPosition() {
        if (actionStartTime > 0) {
            long delay = System.currentTimeMillis() - actionStartTime;
            currentPosition =  actionStatus *  ((double)delay  / maxTimeMillis) * 100 + actionStartPosition;
            // 誤差を丸めて規定範囲内に収める
            if (currentPosition <= 0) {
                currentPosition = 0;
            } else if (currentPosition >= 100) {
                currentPosition = 100;
            }
            if (logger.isDebugEnabled()) {
                logger.debug("actionStart=" + actionStartTime + ", delay=" + delay
                        + ", start_pos=" + Math.round((float)actionStartPosition)
                        + ", current_pos=" + Math.round((float)currentPosition));
            }
            setValue(Math.round((float)currentPosition));
        }
    }

    /**
     * 内部保持している、正方向制御入力ピンを取得します。
     * @return 制御入力ピン
     */
    GpioPinDigitalInput getGpioUpInputPin() {
        return this.gpioPinUpInput;
    }

    /**
     * 内部保持している、負方向制御入力ピンを取得します。
     * @return 制御入力ピン
     */
    GpioPinDigitalInput getGpioDownInputPin() {
        return this.gpioPinDownInput;
    }

    /*
     * 実際の機器を正方向動作させる処理
     */
    private void doUpPosition() {
        if (gpioPinDownOutput != null) {
            gpioPinDownOutput.setState(isInverse());
        }
        if (gpioPinUpOutput != null) {
            gpioPinUpOutput.setState(!isInverse());
        }
    }

    /*
     * 実際の機器を負方向動作させる処理
     */
    private void doDownPosition() {
        if (gpioPinUpOutput != null) {
            gpioPinUpOutput.setState(isInverse());
        }
        if (gpioPinDownOutput != null) {
            gpioPinDownOutput.setState(!isInverse());
        }
    }

    /*
     * 実際の機器を停止させる処理
     */
    private void doStopPosition() {
        if (gpioPinUpOutput != null) {
            gpioPinUpOutput.setState(isInverse());
        }
        if (gpioPinDownOutput != null) {
            gpioPinDownOutput.setState(isInverse());
        }
    }


}
