/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.pi;

import java.sql.SQLException;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.wabit.uecs.pi.db.DatabaseUtils;
import com.wabit.uecs.pi.db.LogEntity;
import com.wabit.uecs.pi.util.MessageUtils;

/**
 * DBを格納場所としてログ情報を読み書きするユーティリティです。
 *
 * @author WaBit
 */
public class UecsPiLogger {

    /** カテゴリー：ノード */
    public static final String CATEGORY_NODE = "NODE";
    /** カテゴリー：デバイス */
    public static final String CATEGORY_DEVICE = "DEVICE";
    /** カテゴリー：その他 */
    public static final String CATEGORY_OTHER = "OTHER";

    // 内部ロガー
    private static Log logger = LogFactory.getLog(UecsPiLogger.class);

    private UecsPiLogger() {

    }

    /**
     * ログ一覧を取得します。
     * @param fetchSize 最大取得件数
     * @return 最新順のログリスト
     */
    public static List<LogEntity> getLogs(int fetchSize) {
        try {
            return DatabaseUtils.getLogs(fetchSize);
        } catch (SQLException e) {
            logger.error(e);
        }
        return Collections.emptyList();
    }

    /**
     * カテゴリを指定してログ一覧を取得します。
     * @param category カテゴリ
     * @param fetchSize 最大取得件数
     * @return 最新順のログリスト
     */
    public static List<LogEntity> getLogs(String category, int fetchSize) {
        try {
            return DatabaseUtils.getLogs(category, fetchSize);
        } catch (SQLException e) {
            logger.error(e);
        }
        return Collections.emptyList();
    }

    /**
     * ログを記録します。
     * @param category カテゴリ
     * @param messageCode メッセージコード
     * @param args メッセージ埋め込みパラメータ
     */
    public static void log(String category, String messageCode, Object... args) {
        String message = MessageUtils.getMessage(messageCode, args);
        try {
            LogEntity entity = new LogEntity();
            entity.setTime(new Date());
            entity.setCategory(category);
            entity.setMessage(message);
            DatabaseUtils.saveLog(entity);
        } catch (SQLException e) {
            logger.error(e);
        }
    }

    /**
     * ログを前削除します。
     */
    public static void clearLogs() {
        try {
            DatabaseUtils.clearLogs();
        } catch (SQLException e) {
            logger.error(e);
        }
    }


}
