/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.pi.webui;

import java.util.Locale;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.wicket.Page;
import org.apache.wicket.RuntimeConfigurationType;
import org.apache.wicket.Session;
import org.apache.wicket.authroles.authentication.AbstractAuthenticatedWebSession;
import org.apache.wicket.authroles.authentication.AuthenticatedWebApplication;
import org.apache.wicket.markup.html.WebPage;
import org.apache.wicket.markup.html.panel.Panel;
import org.apache.wicket.request.Request;
import org.apache.wicket.request.Response;

import com.wabit.uecs.pi.AppConstants;
import com.wabit.uecs.pi.UecsPiNode;
import com.wabit.uecs.pi.UecsPiNodeConfig;
import com.wabit.uecs.pi.db.DatabaseManager;
import com.wabit.uecs.pi.util.MessageUtils;
import com.wabit.uecs.pi.webui.signin.SignInPage;
import com.wabit.uecs.pi.webui.top.TopPage;

/**
 * UECS-PiのWebUI機能を統括するWicketアプリケーションクラスです。
 * 独自のWebUIアプリケーションを作成する場合は、本クラスを継承した子クラスを作成します。
 *
 * @author WaBit
 *
 */
public class WebUIApplication extends AuthenticatedWebApplication {
    // シングルトンインスタンス
    private static UecsPiNode instance;
    // 言語ケース
    private Locale defaultLocale = Locale.getDefault();

    private Log logger = LogFactory.getLog(getClass());

    /**
     * UECSノードインスタンスを生成します。
     * @return UECSノード
     */
    protected UecsPiNode createNodeInstance() {
        return new UecsPiNode();
    }

    /**
     * UECSノードインスタンスを取得します。
     * @return シングルトンインスタンス
     */
    public static UecsPiNode getNodeInstance() {
        return instance;
    }

    /**
     * デフォルトトップページを取得します。
     */
    @Override
    public Class<? extends Page> getHomePage() {
        return TopPage.class;
    }

    @Override
    public void init() {
        super.init();
        getMarkupSettings().setStripWicketTags(true);
        String path = getServletContext().getRealPath(AppConstants.DATABASE_PATH);
        try {
            DatabaseManager.init(path);
            instance = createNodeInstance();
            instance.setDevelopmentMode(isDevelopmentMode());
            instance.restart();

            String locale = instance.getConfig().getString(UecsPiNodeConfig.KEY_LOCALE);
            if (locale != null) {
                Locale lc = Locale.forLanguageTag(locale);
                if (lc != null) {
                    setDefaultLocale(lc);
                }
            }

        } catch (Throwable t) {
            logger.error("init error.", t);
        } finally {

        }
    }

    /**
     * 開発モードであるかどうかを返します。
     * @return 開発モードの場合true
     */
    public static boolean isDevelopmentMode() {
        return WebUIApplication.get().getConfigurationType() == RuntimeConfigurationType.DEVELOPMENT;
    }


    /**
     * アプリケーション終了時処理を行います。
     */
    @Override
    public void onDestroy() {
        getNodeInstance().stop();
        getNodeInstance().onDestroy();
        super.onDestroy();
    }

    /**
     * ロケールを設定値に変更したセッションを取得します。
     */
    @Override
    public Session newSession( Request request, Response response ) {
        Session session = super.newSession(request, response);
        if (getDefaultLocale() != null) {
            session.setLocale(getDefaultLocale());
        }
        return session;
    }


    /**
     * デフォルト言語ロケールを取得します。
     * @return ロケール
     */
    public Locale getDefaultLocale() {
        return defaultLocale;
    }

    /**
     * デフォルト言語ロケールを設定します。
     * @param newLocale
     */
    public void setDefaultLocale(Locale newLocale) {
        this.defaultLocale = newLocale;
        MessageUtils.changeLocale(defaultLocale);
    }

    /**
     * 選択可能な言語ロケールを取得します。
     * @return ロケールリスト
     */
    public Locale[] getAvailableLocales() {
        return new Locale[]{Locale.US, Locale.JAPAN};
    }

    /**
     * ログイン画面ページクラスを取得します。
     * オーバーライドして独自のクラスを返すように変更するとカスタマイズすることができます。
     *
     * @return ログイン画面クラス
     */
    @Override
    protected Class<? extends WebPage> getSignInPageClass() {
        return SignInPage.class;
    }

    /**
     * ログイン処理クラスを取得します。
     * オーバーライドして独自のクラスを返すように変更するとカスタマイズすることができます。
     *
     * @return ログイン処理クラス
     */
    @Override
    protected Class<? extends AbstractAuthenticatedWebSession> getWebSessionClass() {
        return AuthSession.class;
    }

    /**
     * ヘッダー部分表示クラスを取得します。
     * オーバーライドして独自のクラスを返すように変更するとカスタマイズすることができます。
     *
     * @return ヘッダーパネルクラス
     */
    protected Class<? extends Panel> getHeaderPanelClass() {
        return HeaderPanel.class;
    }

    /**
     * メニュー部分表示クラスを取得します。
     * オーバーライドして独自のクラスを返すように変更するとカスタマイズすることができます。
     *
     * @return メニューパネルクラス
     */
    protected Class<? extends Panel> getMenuPanelClass() {
        return MenuPanel.class;
    }

    /**
     * フッター部分表示クラスを取得します。
     * オーバーライドして独自のクラスを返すように変更するとカスタマイズすることができます。
     *
     * @return フッターパネルクラス
     */
    protected Class<? extends Panel> getFooterPanelClass() {
        return FooterPanel.class;
    }

    /**
     * メッセージ表示クラスを取得します。
     * オーバーライドして独自のクラスを返すように変更するとカスタマイズすることができます。
     *
     * @return メッセージパネル
     */
    protected Class<? extends Panel> getFeedbackPanelClass() {
        return CustomFeedbackPanel.class;
    }

}
