/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.pi.db;

import java.io.File;
import java.sql.SQLException;
import java.util.concurrent.Callable;

import com.j256.ormlite.dao.Dao;
import com.j256.ormlite.dao.DaoManager;
import com.j256.ormlite.jdbc.JdbcConnectionSource;
import com.j256.ormlite.misc.TransactionManager;
import com.j256.ormlite.support.ConnectionSource;
import com.j256.ormlite.table.TableUtils;

/**
 * DBトランザクション処理やDAO管理を行うクラスです。
 *
 * @author WaBit
 */
public class DatabaseManager {

    private static String dataBaseURL;
    private static JdbcConnectionSource source;

    /**
     * 初期化を行います。
     * @param dbPath DBファイルパス
     * @throws SQLException 初期化に失敗するとスローされます。
     */
    public static void init(String dbPath) throws SQLException  {
        dataBaseURL = "jdbc:sqlite:" + dbPath;
        source = new JdbcConnectionSource(dataBaseURL);
        File dbFile = new File(dbPath);
        if (!dbFile.exists()) {
            // 初期DB作成
            dbFile.getParentFile().mkdirs();
            createTables(new Class<?>[]{NodeConfigEntity.class, ComponentValueEntity.class, DeviceEntity.class
                    , DeviceConfigEntity.class, ComponentEntity.class, ComponentConfigEntity.class
                    , LogEntity.class});
        }
    }

    /**
     * コネクションを取得します。
     * @return DBコネクション
     */
    private static ConnectionSource getConnectionSource() {
        return source;
    }

    /**
     * トランザクション制御実行します。
     * @param callable トランザクション内処理が実装されたオブジェクト
     * @throws SQLException 処理が失敗するとスローされます。
     */
    public static synchronized void callInTransaction(Callable<?> callable) throws SQLException {
        TransactionManager.callInTransaction(DatabaseManager.getConnectionSource(), callable);
    }

    /**
     * DAOを作成します。
     * @param clazz エンティティクラス
     * @return DAOインスタンス
     * @throws Exception 取得に失敗するとスローされます。
     */
    public synchronized static <T, ID> Dao<T, ID> createDao(Class<T> clazz) throws SQLException {
        Dao<T, ID> dao = DaoManager.createDao(source, clazz);
        //dao.setAutoCommit(source.getReadWriteConnection(), false);
        return dao;
    }


    /**
     * テーブルを作成します。
     * @param tables テーブルリスト
     * @throws SQLException 作成に失敗するとスローされます。
     */
    public static void createTables(Class<?>... tables) throws SQLException {
        for (Class<?> table : tables) {
            TableUtils.createTable(source, table);
        }
    }

}
