/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.device.actuator;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * 一定間隔で繰り返し実行を行うアクションクラス
 *
 * @author WaBit
 */
public abstract class AbstractCyclicAction implements ICyclicAction {

    // 初回実行待ち時間
    private long initialDelay;

    // チェック間隔
    private long interval;
    // 実行スレッドサービス
    private ScheduledExecutorService executor;

    /*
     * 実行タスク
     */
    private class Task implements Runnable {
        private ActuatorComponent<?> comp;

        public Task(ActuatorComponent<?> dev) {
            this.comp = dev;
        }

        @Override
        public void run() {

            try {
                execute(comp);
            } catch (Exception e) {
                if (comp instanceof ActuatorComponent) {
                    ((ActuatorComponent<?>)comp).notifyException(e);
                } else {
                    e.printStackTrace();
                }
            }

        }
    };

    /**
     * 動作間隔を指定したコンストラクタです。
     * @param checkInterval 動作間隔(msec)
     */
    public AbstractCyclicAction(long checkInterval) {
        this.interval = checkInterval;
    }

    /**
     * 動作間隔と初回待ち時間を指定したコンストラクタです。
     * @param checkInterval 動作間隔(msec)
     * @param initialDelay 初回実行待ち時間(msec)
     */
    public AbstractCyclicAction(long checkInterval, long initialDelay) {
        this.interval = checkInterval;
        this.initialDelay = initialDelay;
    }

    /*
     * (non-Javadoc)
     * @see jp.co.toyoct.uecs.device.ICyclicAction#getInterval()
     */
    @Override
    public long getInterval() {
        return interval;
    }

    /*
     * (non-Javadoc)
     * @see com.wabit.uecs.device.actuator.IActuatorAction#onStart(com.wabit.uecs.device.actuator.ActuatorComponent)
     */
    @Override
    public void onStart(ActuatorComponent<?> component) throws Exception {
        if (executor == null) {
            executor = Executors.newSingleThreadScheduledExecutor();
        }

        Task task = new Task(component);
        executor.scheduleAtFixedRate(task, initialDelay, getInterval(), TimeUnit.MILLISECONDS);
    }

    /*
     * (non-Javadoc)
     * @see com.wabit.uecs.device.actuator.IActuatorAction#onStop(com.wabit.uecs.device.actuator.ActuatorComponent)
     */
    @Override
    public void onStop(ActuatorComponent<?> component) throws Exception {
        if (executor != null && !executor.isShutdown()) {
            executor.shutdown();
            executor = null;
        }
    }

}
