/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.pi.webui;

import java.util.ArrayList;
import java.util.List;

import org.apache.wicket.markup.html.form.ChoiceRenderer;
import org.apache.wicket.markup.html.form.DropDownChoice;
import org.apache.wicket.model.Model;
import org.apache.wicket.util.io.IClusterable;

import com.wabit.uecs.device.ComponentConfig;
import com.wabit.uecs.device.IComponent;
import com.wabit.uecs.device.IDevice;

/**
 * コンポーネント選択用共通パーツです。
 *
 * @author WaBit
 *
 */
public class ComponentChoice extends DropDownChoice<ComponentChoice.Row> {

    /** 新規登録用のダミーID定数です */
    public static final String NEW_ID = "NewComponent";

    private Class<? extends IComponent<?>> componentCls;

    public static final class Row implements IClusterable {
        private String id;
        private String name;

        public Row(String id, String name) {
            this.id = id;
            this.name = name;
        }

        public String getId() {
            return id;
        }

        public void setId(String id) {
            this.id = id;
        }

        public String getName() {
            return name;
        }

        public void setName(String name) {
            this.name = name;
        }

    }

    private boolean containsNew;

    /**
     * IComponent型の選択リストを生成します。
     * @param id 画面コンポーネントID
     * @param compoCls コンポーネントクラス
     */
    public <U extends IComponent<?>> ComponentChoice(String id, Class<U> compoCls) {
        this(id, compoCls, true);
    }

    /**
     * IComponent型の選択リストを生成します。
     * @param id 画面コンポーネントID
     * @param compoCls コンポーネントクラス
     * @param containsNew 新規デバイス行有効フラグ
     */
    public <U extends IComponent<?>> ComponentChoice(String id, Class<U> compoCls, boolean containsNew) {
        super(id);
        setChoiceRenderer(new ChoiceRenderer<Row>("Name", "Id"));
        this.componentCls = compoCls;
        this.containsNew = containsNew;
    }

    /**
     * 内部リストを初期化します。
     * @param deviceId 対象とするデバイスID
     * @param componentId 選択状態にするコンポーネントID
     */
    public void init(String deviceId, String componentId) {
        IDevice<?> device = WebUIApplication.getNodeInstance().getDevice(deviceId);
        if (device == null) {
            return;
        }
        List<? extends IComponent<?>> components = device.listComponents(componentCls);
        List<Row> rows = new ArrayList<Row>(components.size() + 1);
        if (containsNew) {
            Row row = new Row(NEW_ID, NEW_ID);
            rows.add(row);
        }
        for (IComponent<?> dev : components) {
            Row row = new Row(dev.getId(), dev.getConfig().getString(ComponentConfig.KEY_COMPONENT_NAME, ""));
            rows.add(row);
        }
        setChoices(rows);
        if (rows.size() > 0) {
            if (componentId == null || NEW_ID.equals(componentId)) {
                setModel(Model.of(rows.get(0)));
            } else {
                for (Row r : rows) {
                    if (r.getId().equals(componentId)) {
                        setModel(Model.of(r));
                        break;
                    }
                }
            }
        }
    }

    @Override
    protected boolean localizeDisplayValues() {
        return true;
    }

}
