package com.wabit.uecs.pi.device.gpio;

import java.util.Arrays;
import java.util.List;

import com.pi4j.io.gpio.GpioController;
import com.pi4j.io.gpio.GpioFactory;
import com.pi4j.io.gpio.GpioPin;
import com.pi4j.io.gpio.GpioPinAnalogInput;
import com.pi4j.io.gpio.GpioPinAnalogOutput;
import com.pi4j.io.gpio.GpioPinDigitalInput;
import com.pi4j.io.gpio.GpioPinDigitalOutput;
import com.pi4j.io.gpio.GpioPinInput;
import com.pi4j.io.gpio.GpioPinPwmOutput;
import com.pi4j.io.gpio.Pin;
import com.pi4j.io.gpio.PinPullResistance;
import com.pi4j.io.gpio.PinState;
import com.pi4j.io.gpio.RaspiPin;
import com.pi4j.io.gpio.event.GpioPinListener;
import com.wabit.uecs.pi.AppConstants;
import com.wabit.uecs.pi.UecsPiNode;
import com.wabit.uecs.pi.UecsPiNodeConfig;
import com.wabit.uecs.pi.webui.WebUIApplication;


/**
 * Raspberry Pi標準GPIO端子動作デバイスクラスです.
 * @author WaBit
 *
 * @param <T>
 */
public class RaspiGpioDevice<T extends GpioDeviceConfig> extends GpioDeviceBase<T> {


    // 全ピン
    private static final Pin[] RASPI_PINS_ALL = new Pin[] {
        RaspiPin.GPIO_00, RaspiPin.GPIO_01, RaspiPin.GPIO_02, RaspiPin.GPIO_03, RaspiPin.GPIO_04, RaspiPin.GPIO_05,
        RaspiPin.GPIO_06, RaspiPin.GPIO_07, RaspiPin.GPIO_08, RaspiPin.GPIO_09, RaspiPin.GPIO_10, RaspiPin.GPIO_11,
        RaspiPin.GPIO_12, RaspiPin.GPIO_13, RaspiPin.GPIO_14, RaspiPin.GPIO_15, RaspiPin.GPIO_16, RaspiPin.GPIO_17,
        RaspiPin.GPIO_18, RaspiPin.GPIO_19, RaspiPin.GPIO_20, RaspiPin.GPIO_21, RaspiPin.GPIO_22, RaspiPin.GPIO_23,
        RaspiPin.GPIO_24, RaspiPin.GPIO_25, RaspiPin.GPIO_26, RaspiPin.GPIO_27, RaspiPin.GPIO_28, RaspiPin.GPIO_29
    };

    // ModelB用ピン
    private static final Pin[] RASPI_PINS_MODEL_B = new Pin[] {
        RaspiPin.GPIO_00, RaspiPin.GPIO_01, RaspiPin.GPIO_02, RaspiPin.GPIO_03,
        RaspiPin.GPIO_04, RaspiPin.GPIO_05, RaspiPin.GPIO_06, RaspiPin.GPIO_07,
        RaspiPin.GPIO_17,
        RaspiPin.GPIO_18, RaspiPin.GPIO_19, RaspiPin.GPIO_20
    };

    // ModelB+/Rspi2 用ピン
    private static final Pin[] RASPI_PINS_MODEL_B_PLUS = new Pin[] {
        RaspiPin.GPIO_00, RaspiPin.GPIO_01, RaspiPin.GPIO_02, RaspiPin.GPIO_03,
        RaspiPin.GPIO_04, RaspiPin.GPIO_05, RaspiPin.GPIO_06, RaspiPin.GPIO_07,
        RaspiPin.GPIO_21,
        RaspiPin.GPIO_22, RaspiPin.GPIO_23, RaspiPin.GPIO_24, RaspiPin.GPIO_25,
        RaspiPin.GPIO_26, RaspiPin.GPIO_27, RaspiPin.GPIO_28, RaspiPin.GPIO_29
    };

    private GpioController gpioController;

    protected RaspiGpioDevice(String deviceId, T config) {
        super(deviceId, config);
    }

    /*
     * (非 Javadoc)
     * @see com.wabit.uecs.pi.device.gpio.GpioDeviceBase#onInit()
     */
    @Override
    protected void onInit() throws Exception {
        super.onInit();
    }

    @Override
    protected void onStart() throws Exception {
        super.onStart();
        // 本番モードのときだけ実際のGPIO操作
        if (!WebUIApplication.getNodeInstance().isDevelopmentMode()) {
            gpioController = GpioFactory.getInstance();
        }
    };

    @Override
    protected void onStop() throws Exception {
        // 本番モードのときだけ実際のGPIO操作
        if (!WebUIApplication.getNodeInstance().isDevelopmentMode()) {
            //シャットダウンすると、ノード再起動後に使えなくなるので行わない
            gpioController.shutdown();
        }
        super.onStop();
    }

    /*
     * (非 Javadoc)
     * @see com.wabit.uecs.pi.device.gpio.GpioDeviceBase#getPin(java.lang.String)
     */
    @Override
    public Pin getPin(String pinName) {
        for (Pin pin : RASPI_PINS_ALL) {
            if (pin.getName().equals(pinName)) {
                return pin;
            }
        }
        return null;
    }


    /**
     * 制御ピンを取得します。
     * @return
     */
    public List<Pin> getControlPins() {
        UecsPiNode node = WebUIApplication.getNodeInstance();
        String model = node.getConfig().getString(UecsPiNodeConfig.KEY_RASPI_MODEL, AppConstants.RASPI_MODEL_B);
        List<Pin> pins = null;
        if (AppConstants.RASPI_MODEL_B.equals(model)) {
            pins = Arrays.asList(RASPI_PINS_MODEL_B);
        } else {
            pins = Arrays.asList(RASPI_PINS_MODEL_B_PLUS);
        }
        return pins;
    }

    /*
     * (非 Javadoc)
     * @see com.wabit.uecs.pi.device.gpio.GpioDeviceBase#getProvisionedGpioPinInput(com.pi4j.io.gpio.Pin)
     */
    @Override
    public GpioPinDigitalInput getProvisionedGpioPinInput(Pin pin) {
        for (GpioPin gpioPin : gpioController.getProvisionedPins()) {
            if (gpioPin.getPin().equals(pin) && gpioPin instanceof GpioPinDigitalInput) {
                return (GpioPinDigitalInput)gpioPin;
            }
        }
        return null;
    }

    /*
     * (非 Javadoc)
     * @see com.wabit.uecs.pi.device.gpio.GpioDeviceBase#getProvisionedGpioPinOutput(com.pi4j.io.gpio.Pin)
     */
    @Override
    public GpioPinDigitalOutput getProvisionedGpioPinOutput(Pin pin) {
        for (GpioPin gpioPin : gpioController.getProvisionedPins()) {
            if (gpioPin.getPin().equals(pin) && gpioPin instanceof GpioPinDigitalOutput) {
                return (GpioPinDigitalOutput)gpioPin;
            }
        }
        return null;
    }

    /*
     * (非 Javadoc)
     * @see com.wabit.uecs.pi.device.gpio.GpioDeviceBase#addListener(com.pi4j.io.gpio.event.GpioPinListener, com.pi4j.io.gpio.GpioPinInput)
     */
    @Override
    public void addListener(GpioPinListener listener, GpioPinInput pin) {
        gpioController.addListener(listener, pin);
    }

    /*
     * (非 Javadoc)
     * @see com.wabit.uecs.pi.device.gpio.GpioDeviceBase#removeListener(com.pi4j.io.gpio.event.GpioPinListener, com.pi4j.io.gpio.GpioPinInput)
     */
    @Override
    public void removeListener(GpioPinListener listener, GpioPinInput pin) {
        gpioController.removeListener(listener, pin);
    }

    /*
     * (非 Javadoc)
     * @see com.wabit.uecs.pi.device.gpio.GpioDeviceBase#provisionDigitalInputPin(com.pi4j.io.gpio.Pin, com.pi4j.io.gpio.PinPullResistance)
     */
    @Override
    public GpioPinDigitalInput provisionDigitalInputPin(Pin pin, PinPullResistance registance) {
        return gpioController.provisionDigitalInputPin(pin, registance);
    }

    /*
     * (非 Javadoc)
     * @see com.wabit.uecs.pi.device.gpio.GpioDeviceBase#provisionDigitalOutputPin(com.pi4j.io.gpio.Pin, com.pi4j.io.gpio.PinState)
     */
    @Override
    public GpioPinDigitalOutput provisionDigitalOutputPin(Pin pin, PinState state) {
        return gpioController.provisionDigitalOutputPin(pin, state);
    }

    /*
     * (非 Javadoc)
     * @see com.wabit.uecs.pi.device.gpio.GpioDeviceBase#provisionAnalogInputPin(com.pi4j.io.gpio.Pin)
     */
    @Override
    public GpioPinAnalogInput provisionAnalogInputPin(Pin pin) {
        return gpioController.provisionAnalogInputPin(pin);
    }

    /*
     * (非 Javadoc)
     * @see com.wabit.uecs.pi.device.gpio.GpioDeviceBase#provisionAnalogOutputPin(com.pi4j.io.gpio.Pin)
     */
    @Override
    public GpioPinAnalogOutput provisionAnalogOutputPin(Pin pin) {
        return gpioController.provisionAnalogOutputPin(pin);
    }

    /*
     * (非 Javadoc)
     * @see com.wabit.uecs.pi.device.gpio.GpioDeviceBase#provisionPwmOutputPin(com.pi4j.io.gpio.Pin)
     */
    @Override
    public GpioPinPwmOutput provisionPwmOutputPin(Pin pin) {
        return gpioController.provisionPwmOutputPin(pin);
    }

    /*
     * (非 Javadoc)
     * @see com.wabit.uecs.pi.device.gpio.GpioDeviceBase#unprovisionPin(com.pi4j.io.gpio.GpioPin[])
     */
    @Override
    public void unprovisionPin(GpioPin... paramVarArgs) {
        gpioController.unprovisionPin(paramVarArgs);
    }


}
