/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.pi.device.gpio;

import java.util.concurrent.Callable;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.j256.ormlite.dao.Dao;
import com.pi4j.io.gpio.GpioPinDigitalInput;
import com.pi4j.io.gpio.GpioPinDigitalOutput;
import com.pi4j.io.gpio.Pin;
import com.pi4j.io.gpio.PinPullResistance;
import com.pi4j.io.gpio.PinState;
import com.wabit.uecs.ActionMode;
import com.wabit.uecs.device.actuator.FixedValueAction;
import com.wabit.uecs.pi.db.ComponentConfigEntity;
import com.wabit.uecs.pi.db.ComponentEntity;
import com.wabit.uecs.pi.db.DatabaseManager;
import com.wabit.uecs.pi.device.ISwitcher;
import com.wabit.uecs.pi.webui.WebUIApplication;

/**
 * リレー制御を想定したスイッチアクチュエータの実装クラスです。
 *
 * @author WaBit
 */
public class DigitalSwitchActuator<T extends DigitalSwitchActuatorConfig> extends GpioActuatorBase<T> implements ISwitcher {

    private GpioPinDigitalOutput gpioPinSwOutput;
    private GpioPinDigitalInput gpioPinSwInput;
    private Log logger = LogFactory.getLog(getClass());

    /**
     * コンストラクタです。
     * @param id コンポーネントID
     * @param config 設定値
     */
    public DigitalSwitchActuator(String id, T config) {
        super(id, config);
    }

    /*
     * (非 Javadoc)
     * @see com.wabit.uecs.pi.device.ISwitcher#on()
     */
    @Override
    public void on() {
        setValue(ON);
    }

    /*
     * (非 Javadoc)
     * @see com.wabit.uecs.pi.device.ISwitcher#off()
     */
    @Override
    public void off() {
        setValue(OFF);
    }

    /*
     * (非 Javadoc)
     * @see com.wabit.uecs.pi.device.ISwitcher#isOn()
     */
    @Override
    public boolean isOn() {
        return (getValue() != null && getValue().intValue() == ON) ? true : false;
    }

    /*
     * (非 Javadoc)
     * @see com.wabit.uecs.pi.device.ISwitcher#isOff()
     */
    @Override
    public boolean isOff() {
        return (getValue() == null || getValue().intValue() == OFF) ? true : false;
    }

    /**
     * スイッチ制御出力用ピンを取得します。
     * @return ピンインスタンス
     */
    public Pin getSwitchOutputPin() {
        String pinName = getConfig().getString(DigitalSwitchActuatorConfig.KEY_GPIO_PIN_SW_OUT);
        if (pinName != null && !pinName.isEmpty()) {
            return ((GpioDeviceBase<?>) getDevice()).getPin(pinName);
        }
        return null;
    }

    /**
     * スイッチ制御入力用ピンを取得します。
     * @return ピンインスタンス
     */
    public Pin getSwitchInputPin() {
        String pinName = getConfig().getString(DigitalSwitchActuatorConfig.KEY_GPIO_PIN_SW_IN);
        if (pinName != null && !pinName.isEmpty()) {
            return ((GpioDeviceBase<?>) getDevice()).getPin(pinName);
        }
        return null;
    }

    /**
     * GPIOピンとしてのH/L状態を取得します。(H/L反転設定を考慮)
     * @param val 値
     * @return H=true, L=false
     */
    protected boolean getState(double val) {
        boolean state = false;
        if (val >= 1.0) {
            state = true;
        }
        if (isInverse()) {
            return !state;
        }
        return state;
    }

    /**
     * GPIOピンを初期化し、rcM/rcA受信動作アクションを登録します。
     */
    @Override
    protected void onInit() throws Exception {
        super.onInit();

        // インターロック動作
        setAction(ActionMode.Interlock, new DigitalSwitchInterlockAction());

        // スタンドアローンモード動作
        FixedValueAction stdAction = new FixedValueAction();
        stdAction.setValue(getConfig().getInt(DigitalSwitchActuatorConfig.KEY_DEFAULT_VALUE, 0));
        setAction(ActionMode.Standalone, stdAction);

    }

    @Override
    protected void onStart() throws Exception {

        GpioDeviceBase<?> device = (GpioDeviceBase<?>)getDevice();

        // 本番モードのときだけ実際のGPIO初期化
        if (!WebUIApplication.getNodeInstance().isDevelopmentMode()) {

            // 制御出力ピンは必須
            Pin pin = getSwitchOutputPin();
            if (pin != null && gpioPinSwOutput == null) {
                boolean state = getState(getValue().doubleValue());
                gpioPinSwOutput = device.provisionDigitalOutputPin(pin, PinState.getState(state));
                if (gpioPinSwOutput == null) {
                    throw new Exception("GPIO pin not found : " + pin);
                }
            }
            logger.debug("switch_pin(OUTPUT) : " + gpioPinSwOutput);

            // 制御入力は任意
            pin = getSwitchInputPin();
            if (pin != null && gpioPinSwInput == null) {
                gpioPinSwInput = device.provisionDigitalInputPin(pin, PinPullResistance.PULL_UP);
                if (gpioPinSwInput == null) {
                    throw new Exception("GPIO pin not found : " + pin);
                }
                logger.debug("switch_pin(INPUT) : " + gpioPinSwInput);
            }
        }

        super.onStart();
    }

    @Override
    public void setValue(Number value) {
        if (gpioPinSwOutput != null && value != null) {
            ((GpioPinDigitalOutput) gpioPinSwOutput).setState(getState(value.doubleValue()));
        }
        super.setValue(value);
    }

    /**
     * 最終位置をDBに保存します。
     */
    @Override
    protected void onStop() throws Exception {

        if (getValue() != null) {
            DatabaseManager.callInTransaction(new Callable<Void>() {
                public Void call() throws Exception {
                    Dao<ComponentEntity, String> compDao = DatabaseManager.createDao(ComponentEntity.class);
                    if (compDao.queryForId(getId()) != null) {
                        Dao<ComponentConfigEntity, String> confDao = DatabaseManager
                                .createDao(ComponentConfigEntity.class);
                        ComponentConfigEntity ent = new ComponentConfigEntity();
                        ent.setComponentId(getId());
                        ent.setId(getId() + "." + DigitalSwitchActuatorConfig.KEY_LAST_VALUE);
                        ent.setKey(DigitalSwitchActuatorConfig.KEY_LAST_VALUE);
                        ent.setValue(getValue().toString());
                        confDao.createOrUpdate(ent);
                    }
                    return null;
                }
            });
        }

        GpioDeviceBase<?> device = (GpioDeviceBase<?>)getDevice();
        if (gpioPinSwOutput != null) {
            device.unprovisionPin(gpioPinSwOutput);
            gpioPinSwOutput = null;
        }

        if (gpioPinSwInput != null) {
            device.unprovisionPin(gpioPinSwInput);
            gpioPinSwInput = null;
        }
        super.onStop();
    }

    /**
     * 内部保持している、制御入力ピンを取得します。
     * @return 制御入力ピン
     */
    GpioPinDigitalInput getGpioInputPin() {
        return this.gpioPinSwInput;
    }

}
