/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.protocol.handler;

import java.util.List;

import com.wabit.uecs.Ccm;
import com.wabit.uecs.CcmService;
import com.wabit.uecs.IUecsNode;
import com.wabit.uecs.UecsConstants;
import com.wabit.uecs.UecsRequest;
import com.wabit.uecs.UecsResponse;
import com.wabit.uecs.protocol.RequestCcm;
import com.wabit.uecs.protocol.SearchCcm;
import com.wabit.uecs.protocol.ServerCcm;
import com.wabit.uecs.protocol.XmlCcm;


/**
 * 提供者サーチ、応答CCMに対する応答処理を行うハンドラーです。
 * UECS通信規約の表3-9「提供者サーチCCM」の補足内容の処理に従います。<br>
 * UECS通信規約の図3-2「送信間隔がレベルBに設定されているデータ送信用CCMの利用プロセス」に従います。<br>
 * <ol>
 * <li>送信タイミングがレベルB</li>
 * <li>type属性が等しく</li>
 * <li>自身に関連する room, region, order の各属性値(全系統向けの0は関係がある)</li>
 * </ol>
 * @author WaBit
 */
public class SearchProtocolHandler extends AbstractProtocolHandler {

    /**
     * コンストラクタ
     * @param node ノードインスタンス
     */
    public SearchProtocolHandler(IUecsNode<?> node) {
        super(node);
    }

    @Override
    public int getPort() {
        return UecsConstants.SEARCH_PORT;
    }

    @Override
    protected void handleCcm(UecsRequest req, UecsResponse res, XmlCcm ccm) throws Exception {
        if (ccm instanceof SearchCcm) {
            handleSearchCcm(req, res, (SearchCcm)ccm);
        } else if (ccm instanceof ServerCcm) {
            handleServerCcm(req, res, (ServerCcm)ccm);
        } else {
            throw new Exception("illegal packet : " + ccm.toString());
        }

    }

    // SEARCH CCM受信処理
    private void handleSearchCcm(UecsRequest req, UecsResponse res, SearchCcm sccm) {
        IUecsNode<?> node = getNode();

        //登録済みCCMと比較
        List<CcmService> list = node.listCcmService();
        for (CcmService sv : list) {
            Ccm dccm = sv.getCcm();
            // 送信側のみ応答する。
            if (dccm.getSide() != UecsConstants.SENDER) {
                continue;
            }
            // レベルBかつ全棟情報または属性が一致した場合に応答する。
            if (dccm.getLevel().getCategory() == 'B'
                && dccm.getType().equals(sccm.getType())
                && (sccm.getRoom() == 0
                        || dccm.getRoom() == sccm.getRoom())
                && (sccm.getRegion() == 0
                        || dccm.getRegion() == sccm.getRegion())
                && (sccm.getOrder() == 0
                        || dccm.getOrder() == sccm.getOrder())
            ) {
                ServerCcm svccm = new ServerCcm();
                svccm.setUecsVersion(dccm.getUecsVersion());
                svccm.setType(dccm.getType());
                svccm.setRoom(dccm.getRoom());
                svccm.setRegion(dccm.getRegion());
                svccm.setOrder(dccm.getOrder());
                svccm.setPriority(dccm.getPriority());
                svccm.setIpAddress(node.getIpAddress());

                // ブロードキャストアドレスに送る
                res.setAddress(node.getBroadcastAddress());
                res.setPort(svccm.getPort());
                res.setData(svccm.toXmlBytes());

            }
        }
    }

    private void handleServerCcm(UecsRequest req, UecsResponse res, ServerCcm sccm) {
        IUecsNode<?> node = getNode();

        //登録済みCCMと比較
        List<CcmService> list = node.listCcmService();
        for (CcmService sv : list) {
            Ccm dccm = sv.getCcm();
            // 受信側のみ応答する。
            if (dccm.getSide() != UecsConstants.RECEIVER) {
                continue;
            }
            // レベルBかつ全棟情報または属性が一致した場合に応答する。
            if (dccm.getLevel().getCategory() == 'B'
                && dccm.getType().equals(sccm.getType())
                && (sccm.getRoom() == 0
                        || dccm.getRoom() == sccm.getRoom())
                && (sccm.getRegion() == 0
                        || dccm.getRegion() == sccm.getRegion())
                && (sccm.getOrder() == 0
                        || dccm.getOrder() == sccm.getOrder())
            ) {
                RequestCcm request = new RequestCcm();
                request.setUecsVersion(dccm.getUecsVersion());
                request.setType(dccm.getType());
                request.setRoom(dccm.getRoom());
                request.setRegion(dccm.getRegion());
                request.setOrder(dccm.getOrder());
                request.setPriority(dccm.getPriority());

                // ユニキャストでデータ要求を送信する
                res.setAddress(req.getAddress());
                res.setPort(request.getPort());
                res.setData(request.toXmlBytes());
            }
        }
    }

}
