/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs;

import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import com.wabit.uecs.protocol.DataCcm;
import com.wabit.uecs.protocol.SearchCcm;

/**
 * CCM管理サービスクラスです。
 * ノード内でCCMデータや状態管理の機能を提供します。
 *
 * @author WaBit
 *
 */
public class CcmService {

    private Ccm rccm;
    private Ccm ccm;
    private long updateTime = Long.MIN_VALUE;
    private Set<ICcmServiceListener> listeners = Collections
            .synchronizedSet(new HashSet<ICcmServiceListener>());
    private IUecsNode<?> node;
    private String name;
    private boolean isExpired = true;
    private boolean isEnable = true;

    /**
     * コンストラクタ.
     *
     * @param value CCM定義情報
     */
    public CcmService(Ccm value) {
        this.ccm = value;
    }

    /**
     * CCMラベル名称を取得します。
     * @return 名称
     */
    public String getName() {
        return name;
    }

    /**
     * CCMラベル名称を設定します。
     * @param name 名称
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * 動作リスナーを追加します。
     *
     * @param listener リスナー
     * @return 登録されるとtrue
     */
    public boolean addListener(ICcmServiceListener listener) {
        return listeners.add(listener);
    }

    /**
     * 内部リスナセットを取得します。
     * @return リスナセット
     */
    Set<ICcmServiceListener> getListeners() {
        return listeners;
    }

    /**
     * 最終受信CCMを取得します。
     *
     * @return 受信されたCCM。未受信の場合はnull。
     */
    public Ccm getAcceptCcm() {
        return rccm;
    }

    /**
     * 定義情報CCMを取得します。
     * @return CCM
     */
    public Ccm getCcm() {
        return this.ccm;
    }

    /**
     * 最終更新時刻を取得します。
     *
     * @return ミリ秒。未設定の場合は、Long.MIN_VALUEが返ります。
     */
    public long getUpdateTime() {
        return updateTime;
    }

    /**
     * 有効期限切れデータであるかを取得します。
     *
     * @return 期限切れであればtrue
     */
    public boolean isExpired() {
        return isExpired;
    }

    /**
     * サービスが有効かどうかを取得します。
     * @return 有効であれば true, 無効であるときはfalse
     */
    public boolean isEnable() {
        return isEnable;
    }

    /**
     * サービスの状態を設定します。
     * @param isActive 有効にするときはtrue,無効はfalse
     */
    public void setEnable(boolean isActive) {
        this.isEnable = isActive;
    }

    /**
     * サービスを開始します。
     * デフォルトでは何も行いません。
     * {@link IUecsNode#start()}内でコールされます。
     * @param node ノード
     * @throws Exception 処理に失敗するとスローされます。
     */
    protected void onStart(IUecsNode<?> node) throws Exception {

    }

    /**
     * サービスを停止します。
     * デフォルトでは何も行いません。
     * {@link IUecsNode#stop()}内でコールされます。
     * @param node ノード
     * @throws Exception 処理に失敗するとスローされます。
     */
    protected void onStop(IUecsNode<?> node) throws Exception {

    }

    /**
     * 動作リスナーを削除します。
     *
     * @param listener 削除対象リスナー
     * @return 削除されるとtrue
     */
    public boolean removeListener(ICcmServiceListener listener) {
        return listeners.remove(listener);
    }

    /**
     * データ要求を行います。 通信レベルがB、かつ受信側の場合のみ動作します。
     */
    public void requestData() {
        if (node == null || ccm.getLevel() == null || ccm.getLevel().getCategory() != UecsConstants.LEVEL_B
                || ccm.getSide() != UecsConstants.RECEIVER) {
            return;
        }
        SearchCcm req = new SearchCcm();
        req.setUecsVersion(ccm.getUecsVersion());
        req.setType(ccm.getType());
        req.setRoom(ccm.getRoom());
        req.setRegion(ccm.getRegion());
        req.setOrder(ccm.getOrder());
        node.sendPacket(node.getBroadcastAddress(), req.getPort(),
                req.toXmlBytes());
    }

    /**
     * 適合受信CCMを設定します。
     *
     * @param rccm 受信したCCM
     */
    public void setAcceptCcm(Ccm rccm) {
        this.rccm = rccm;
        updateValue(rccm.getStringValue());
        // 受信イベント通知
        for (ICcmServiceListener l : listeners) {
            l.ccmReceived(this, ccm);
        }
    }

    /**
     * ノードに登録された場合にコールバックで設定します。
     * @param node 登録先ノード
     */
    void setNode(IUecsNode<?> node) {
        this.node = node;
    }

    /**
     * ノードインスタンスを返します。
     * @return ノードに登録されていない場合はnullが返却されます。
     */
    public IUecsNode<?> getNode() {
        return this.node;
    }

    /**
     * CCMの現在値を更新します。
     * 送信側のCCMで更新する際に使用します。
     * @param value 更新する値
     */
    public void updateValue(Number value) {
        this.ccm.setValue(value);
        update();
    }

    /**
     * CCMの現在値を文字列値で更新します。
     * 受信側のCCMで更新する際に使用します。
     * @param value 更新する値
     */
    public void updateValue(String value) {
        this.ccm.setValue(value);
        update();
    }

    /**
     * 更新時間と初期値を強制設定します。
     * 初期値設定に用いられることを想定しており、更新イベントは発生しません。
     * @param time 設定時刻
     * @param value 設定する値
     */
    public void setValue(long time, Number value) {
        this.ccm.setValue(value);
        this.updateTime = time;
        //有効期限切れをチェック
        if ((System.currentTimeMillis() - time) > ccm.getLevel().getAvailableTime()) {
            this.isExpired = true;
        } else {
            this.isExpired = false;
        }
    }

    /**
     * 値の更新処理プロセスを実行します。
     */
    private void update() {
        this.updateTime = System.currentTimeMillis();
        this.isExpired = false;

        // CCMがセットされていなかったり、null値の場合は何もしない
        if (ccm == null || ccm.getNumberValue() == null) {
            return;
        }

        // ノードが起動状態でCCMが有効のときのみ以下を実行
        if (node != null && node.isActive() && isEnable()) {
            // 更新時データ送信タイプCCMを送信
            if (ccm.getLevel() != null
                    && ccm.getLevel().isAck()
                    && ccm.getSide() == UecsConstants.SENDER) {
                DataCcm dccm = createDataCcm(ccm);
                node.sendPacket(node.getBroadcastAddress(), dccm.getPort(),
                        dccm.toXmlBytes());
                // CCM送信イベント通知
                for (ICcmServiceListener l : listeners) {
                    l.ccmSent(this, ccm);
                }
            }

            for (ICcmServiceListener l : listeners) {
                l.ccmValueChanged(this, ccm);
            }
        }
    }

    /**
     * 有効期限切れをチェックします。
     */
    void checkExpiration() {
        // すでに有効期限切れの場合や一度も受信していない場合はチェックはしない。
        if (!isExpired() && getAcceptCcm() != null) {
            if (updateTime > Long.MIN_VALUE
                    && (System.currentTimeMillis() - updateTime) > ccm.getLevel()
                            .getAvailableTime()) {
                isExpired = true;
                for (ICcmServiceListener l : listeners) {
                    l.ccmExpired(this, ccm);
                }
            }
        }
    }

    /**
     * データCCMを作成します。
     * @param ccm
     * @return データCCM
     */
    DataCcm createDataCcm(Ccm ccm) {
        DataCcm dccm = new DataCcm();
        dccm.setUecsVersion(ccm.getUecsVersion());
        dccm.setType(ccm.getType());
        dccm.setRoom(ccm.getRoom());
        dccm.setRegion(ccm.getRegion());
        dccm.setOrder(ccm.getOrder());
        dccm.setPriority(ccm.getPriority());
        dccm.setIpAddress(node.getIpAddress());
        dccm.setCast(ccm.getCast());
        dccm.setUnit(ccm.getUnit());
        dccm.setValue(ccm.getNumberValue());
        return dccm;
    }


}
