/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.util;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Properties;

/**
 * 各種目的で使われる設定値保持の汎用格納クラスです。
 * 内部では全ての設定値がSting型のプロパティ値に変換されて保持されています。
 * 各種の型変換して取得するためのユーティリティメソッドが用意されています。
 * @author WaBit
 *
 */
public class PropertyConfig extends Properties {

    private static final long serialVersionUID = -3792185783687697715L;

    /**
     * 空のインスタンスを返します。
     */
    public PropertyConfig() {

    }

    /**
     * nullが設定されると、エントリー削除されます。
     * @see java.util.Hashtable#put(java.lang.Object, java.lang.Object)
     */
    @Override
    public Object put(Object key, Object value) {
        if (value == null) {
            return remove(key);
        }
        return super.put(key, value);
    }

    /**
     * コンストラクタ.
     *
     * @param xmlPath 読込XMLファイルパス
     * @throws IOException 読込に失敗するとスローされます。
     */
    public PropertyConfig(String xmlPath) throws IOException {
        InputStream in = new FileInputStream(xmlPath);
        loadFromXML(in);
    }

    /**
     * 設定パラメータをString型で取得します。
     *
     * @param name 設定値キー名
     * @return 設定値
     */
    public String getString(String name) {
        Object val = get(name);
        if (val != null) {
            return val.toString();
        }
        return null;
    }

    /**
     * 設定パラメータをString型で取得します。
     *
     * @param name 設定値キー名
     * @param defaultValue 設定値がなかった場合に返されるデフォルト値
     * @return 設定値
     */
    public String getString(String name, String defaultValue) {
        if (getString(name) == null) {
            return defaultValue;
        }
        return getString(name);
    }

    /**
     * 設定パラメータをshort型で取得します。
     *
     * @param name 設定値キー名
     * @return 設定値
     */
    public short getShort(String name) {
        String val = getString(name);
        return Short.parseShort(val);
    }

    /**
     * 設定パラメータをshort型で取得します。
     *
     * @param name 設定値キー名
     * @param defaultValue 設定値がなかった場合に返されるデフォルト値
     * @return 設定値
     */
    public short getShort(String name, short defaultValue) {
        String val = getString(name);
        if (val == null || val.length() < 1) {
            return defaultValue;
        }
        return getShort(name);
    }

    /**
     * 設定パラメータをint型で取得します。
     *
     * @param name 設定値キー名
     * @return 設定値
     */
    public int getInt(String name) {
        String val = getString(name);
        return Integer.parseInt(val);
    }

    /**
     * 設定パラメータをint型で取得します。
     *
     * @param name 設定値キー名
     * @param defaultValue 設定値がなかった場合に返されるデフォルト値
     * @return 設定値
     */
    public int getInt(String name, int defaultValue) {
        String val = getString(name);
        if (val == null || val.length() < 1) {
            return defaultValue;
        }
        return getInt(name);
    }

    /**
     * 設定パラメータをfloat型で取得します。
     *
     * @param name 設定値キー名
     * @return 設定値
     */
    public float getFloat(String name) {
        String val = getString(name);
        return Float.parseFloat(val);
    }

    /**
     * 設定パラメータをfloat型で取得します。
     *
     * @param name 設定値キー名
     * @param defaultValue 設定値がなかった場合に返されるデフォルト値
     * @return 設定値
     */
    public float getFloat(String name, float defaultValue) {
        String val = getString(name);
        if (val == null || val.length() < 1) {
            return defaultValue;
        }
        return getFloat(name);
    }

    /**
     * 設定パラメータをdouble型で取得します。
     *
     * @param name 設定値キー名
     * @return 設定値
     */
    public double getDouble(String name) {
        String val = getString(name);
        return Double.parseDouble(val);
    }

    /**
     * 設定パラメータをdouble型で取得します。
     *
     * @param name 設定値キー名
     * @param defaultValue 設定値がなかった場合に返されるデフォルト値
     * @return 設定値
     */
    public double getDouble(String name, double defaultValue) {
        String val = getString(name);
        if (val == null || val.length() < 1) {
            return defaultValue;
        }
        return getDouble(name);
    }

    /**
     * 設定パラメータをlong型で取得します。
     *
     * @param name 設定値キー名
     * @return 設定値
     */
    public long getLong(String name) {
        String val = getString(name);
        return Long.parseLong(val);
    }

    /**
     * 設定パラメータをlong型で取得します。
     *
     * @param name 設定値キー名
     * @param defaultValue 設定値がなかった場合に返されるデフォルト値
     * @return 設定値
     */
    public long getLong(String name, long defaultValue) {
        String val = getString(name);
        if (val == null || val.length() < 1) {
            return defaultValue;
        }
        return getLong(name);
    }

    /**
     * 設定パラメータをboolean型で取得します。
     *
     * @param name 設定値キー名
     * @return 設定値
     */
    public boolean getBoolean(String name) {
        String val = getString(name);
        return Boolean.parseBoolean(val);
    }

    /**
     * 設定パラメータをboolean型で取得します。
     *
     * @param name 設定値キー名
     * @param defaultValue 設定値がなかった場合に返されるデフォルト値
     * @return 設定値
     */
    public boolean getBoolean(String name, boolean defaultValue) {
        String val = getString(name);
        if (val == null || val.length() < 1) {
            return defaultValue;
        }
        return getBoolean(name);
    }

    /**
     * 設定パラメータ("yyyyMMdd")をDate型で取得します。
     *
     * @param name 設定値キー名
     * @return 設定値
     */
    public Date getDate(String name) {
        return getDateTime(name, "yyyyMMdd");
    }

    /**
     * 設定パラメータ("yyyyMMddHHmmss")をDate型で取得します。
     *
     * @param name 設定値キー名
     * @return 設定値
     */
    public Date getDateTime(String name) {
        return getDateTime(name, "yyyyMMddHHmmss");
    }

    /**
     * 設定パラメータから完全指定名でクラスを解析して取得します。
     *
     * @param name 設定値キー名
     * @return 設定値 変換に失敗した場合はnullが返ります。
     */
    public Class<?> getClassObject(String name) {
        try {
            return Class.forName(getString(name));
        } catch (ClassNotFoundException e) {
            return null;
        }
    }

    /*
     * 設定パラメータを取得します。
     *
     * @param name 設定値キー名
     *
     * @return 設定値
     */
    private Date getDateTime(String name, String format) {
        String val = getString(name);
        if (val == null) {
            return null;
        }
        if (val.length() > format.length()) {
            val = val.substring(0, format.length());
        }
        SimpleDateFormat fmt = new SimpleDateFormat(format);
        try {
            return fmt.parse(val);
        } catch (ParseException e) {
            return null;
        }
    }

    /**
     * パラメータをString型で設定します。
     *
     * @param name 設定値キー名
     * @param val 設定値
     */
    public void setString(String name, String val) {
        setProperty(name, val);
    }

    /**
     * パラメータをshort型で設定します。
     *
     * @param name 設定値キー名
     * @param val 設定値
     */
    public void setShort(String name, short val) {
        setProperty(name, Short.toString(val));
    }

    /**
     * パラメータをint型で設定します。
     *
     * @param name 設定値キー名
     * @param val 設定値
     */
    public void setInt(String name, int val) {
        setProperty(name, Integer.toString(val));
    }

    /**
     * パラメータをlong型で設定します。
     *
     * @param name 設定値キー名
     * @param val 設定値
     */
    public void setLong(String name, long val) {
        setProperty(name, Long.toString(val));
    }

    /**
     * パラメータをfloat型で設定します。
     *
     * @param name 設定値キー名
     * @param val 設定値
     */
    public void setFloat(String name, float val) {
        setProperty(name, Float.toString(val));
    }

    /**
     * パラメータをdouble型で設定します。
     *
     * @param name 設定値キー名
     * @param val 設定値
     */
    public void setDouble(String name, double val) {
        setProperty(name, Double.toString(val));
    }

    /**
     * パラメータをboolean型で設定します。
     *
     * @param name 設定値キー名
     * @param val 設定値
     */
    public void setBoolean(String name, boolean val) {
        setProperty(name, Boolean.toString(val));
    }

    /**
     * パラメータをDate型のデフォルトフォーマット("yyyyMMddHHmmss")で設定します。
     *
     * @param name 設定値キー名
     * @param val 設定値
     */
    public void setDate(String name, Date val) {
        setDate(name, val, "yyyyMMddHHmmss");
    }

    /**
     * パラメータをDate型でフォーマット指定して設定します。
     *
     * @param name 設定値キー名
     * @param val 設定値
     * @param format 指定フォーマット
     */
    public void setDate(String name, Date val, String format) {
        SimpleDateFormat fmt = new SimpleDateFormat(format);
        setProperty(name, fmt.format(val));
    }

}
