/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.protocol.handler;

import java.util.ArrayList;
import java.util.List;

import com.wabit.uecs.CcmService;
import com.wabit.uecs.IUecsNode;
import com.wabit.uecs.UecsConstants;
import com.wabit.uecs.UecsRequest;
import com.wabit.uecs.UecsResponse;
import com.wabit.uecs.protocol.CcmScanCcm;
import com.wabit.uecs.protocol.CcmScanReplyCcm;
import com.wabit.uecs.protocol.NodeScanCcm;
import com.wabit.uecs.protocol.NodeScanReplyCcm;
import com.wabit.uecs.protocol.XmlCcm;


/**
 * ノードスキャン要求、通信子スキャン要求CCMに対する応答処理を行うハンドラーです。
 * UECS通信規約の表3-12「ノードスキャン応答CCM」の補足内容の処理に従います。<br>
 * UECS通信規約の表3-14「通信子スキャン応答CCM」の補足内容の処理に従います。<br>
 * @author WaBit
 */
public class ScanProtocolHandler extends AbstractProtocolHandler {
    // CCMリスト分割単位（サイズ制限規約を考慮して2にしておく）
    private static final int DIV_UNIT = 2;

    /**
     * コンストラクタ。
     * @param node ノードインスタンス
     */
    public ScanProtocolHandler(IUecsNode<?> node) {
        super(node);
    }

    @Override
    public int getPort() {
        return UecsConstants.SCAN_PORT;
    }

    @Override
    protected void handleCcm(UecsRequest req, UecsResponse res, XmlCcm ccm) throws Exception {
        if (ccm instanceof NodeScanCcm) {
            handleNodeScanCcm(req, res, (NodeScanCcm)ccm);
        } else if (ccm instanceof CcmScanCcm) {
            handleCcmScanCcm(req, res, (CcmScanCcm)ccm);
        } else {
            throw new Exception("illegal packet : " + ccm.toString());
        }
    }

    // NODESCAN CCM受信処理
    private void handleNodeScanCcm(UecsRequest req, UecsResponse res, NodeScanCcm ccm) {
        IUecsNode<?> node = getNode();
        NodeScanReplyCcm reply = new NodeScanReplyCcm();
        reply.setUecsVersion(node.getUecsVersion());
        reply.setUecsID(node.getUecsID());
        reply.setName(node.getName());
        reply.setVender(node.getVender());
        reply.setIpAddress(node.getIpAddress());
        reply.setMacAddress(node.getMacAddress());

        res.setAddress(req.getAddress());
        res.setPort(reply.getPort());
        res.setData(reply.toXmlBytes());
    }

    // CCMSCAN CCM受信処理
    private void handleCcmScanCcm(UecsRequest req, UecsResponse res, CcmScanCcm rCcm) {
        IUecsNode<?> node = getNode();

        // 登録済みCCMを収集
        List<CcmService> defList = new ArrayList<CcmService>();
        for (CcmService ccmSv : node.listCcmService()) {
            //規約では送信側のみ収集するように受け取れるが、ここでは両方返すようにする
        	if (ccmSv.getCcm() != null
        	    //&& ccmSv.getCcm().getSide() == UecsConstants.SENDER
        	    ) {
        		defList.add(ccmSv);
        	}
        }

        // CCM登録が無い場合は何もしない
        if (defList.size() == 0) {
            return;
        }

        // 総ページ算出
        int total = (defList.size() / DIV_UNIT);
        // 端数があれば1ページ追加
        if (defList.size() % DIV_UNIT > 0) {
            total += 1;
        }

        CcmScanReplyCcm reply = new CcmScanReplyCcm();
        reply.setUecsVersion(node.getUecsVersion());
        reply.setPage(rCcm.getPage());
        reply.setTotal(total);

        //送信ページのCCMのみ抽出
        int start = (rCcm.getPage() - 1) * DIV_UNIT;
        int end = start + DIV_UNIT;
        for (int i=start; i<end; i++) {
            if (i < defList.size()) {
                CcmService sv = defList.get(i);
                reply.putCcm(i, sv.getCcm());
            }
        }

        res.setAddress(req.getAddress());
        res.setPort(reply.getPort());
        res.setData(reply.toXmlBytes());
    }
}
