/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.protocol.handler;

import com.wabit.uecs.IUecsNode;
import com.wabit.uecs.IUecsProtocolHandler;
import com.wabit.uecs.UecsRequest;
import com.wabit.uecs.UecsResponse;
import com.wabit.uecs.protocol.XmlCcm;
import com.wabit.uecs.protocol.XmlCcmParser;

/**
 * プロトコル共通実装基底クラスです。
 * @author WaBit
 *
 */
public abstract class AbstractProtocolHandler implements IUecsProtocolHandler {

    private IUecsNode<?> node;
    private XmlCcmParser ccmParser = new XmlCcmParser();

    /**
     * コンストラクタ.
     * @param node ノードインスタンス
     */
    public AbstractProtocolHandler(IUecsNode<?> node) {
        this.node = node;
    }

    /**
     * ノードインスタンスを取得します。
     * @return ノードインスタンス
     */
    public IUecsNode<?> getNode() {
        return node;
    }

    /**
     * データ受信後、CCM変換処理を行い、handleCcm()を呼び出します。
     */
    @Override
    public void handle(UecsRequest req, UecsResponse res) throws Exception {
        XmlCcm ccm = ccmParser.parseBytes(req.getData());
        ccm.setReceivedTime(req.getTime());
        handleCcm(req, res, ccm);
    }


    /**
     * 例外発生時処理を行います。
     * デフォルトではスタックトレースを標準出力に出します。
     */
    @Override
    public void handleException(UecsRequest req, UecsResponse res, Exception exception) {
    	exception.printStackTrace();
    }


    /**
     * CCM受信処理を行います。
     * @param req リクエスト
     * @param res レスポンス
     * @param ccm 受信CCM
     * @throws Exception 処理に失敗した場合スローされます。
     */
    protected abstract void handleCcm(UecsRequest req, UecsResponse res, XmlCcm ccm) throws Exception;
}
