/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.protocol;

import java.net.InetAddress;
import java.net.UnknownHostException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

import com.wabit.uecs.UecsConstants;
import com.wabit.uecs.util.CommonUtils;

/**
 * ノードスキャン応答CCMクラスです。
 * @author WaBit
 *
 */
public class NodeScanReplyCcm extends XmlCcm {

    /** NODEタグ名 */
    public static final String TAG_NODE = "NODE";

    /** NAMEタグ名 */
    public static final String TAG_NAME = "NAME";

    /** VENDERタグ名 */
    public static final String TAG_VENDER = "VENDER";

    /** UECSIDタグ名 */
    public static final String TAG_UECSID = "UECSID";

    /** IPタグ名 */
    public static final String TAG_IP = "IP";

    /** MACタグ名 */
    public static final String TAG_MAC = "MAC";

    private String name;
    private String vender;
    private String uecsID;
    private InetAddress ipAddress;
    private byte[] macAddress;

    /**
     * 送信元ノードアドレス(IP)を取得します。
     * @return IPアドレス
     */
    public InetAddress getIpAddress() {
        return ipAddress;
    }

    /**
     * 送信元ノードアドレス(IP)を設定します。
     * @param address IPアドレス
     */
    public void setIpAddress(InetAddress address) {
        this.ipAddress = address;
    }

    /**
     * ノード名称を取得します。
     * @return 名称
     */
    public String getName() {
        return name;
    }

    /**
     * ベンダー名を取得します。
     * @return ベンダー名
     */
    public String getVender() {
        return vender;
    }

    /**
     * UECS IDを取得します。
     * @return UECS ID
     */
    public String getUecsID() {
        return uecsID;
    }

    /**
     * MACアドレスを取得します。
     * @return MACアドレス
     */
    public byte[] getMacAddress() {
        return macAddress;
    }

    /**
     * ノード名称を設定します
     * @param name ノード名称
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * ベンダー名を設定します。
     * @param vender ベンダー名
     */
    public void setVender(String vender) {
        this.vender = vender;
    }

    /**
     * UECS IDを設定します。
     * @param uecsID UECS ID
     */
    public void setUecsID(String uecsID) {
        this.uecsID = uecsID;
    }

    /**
     * MACアドレスを設定します。
     * @param macAddress MACアドレス
     */
    public void setMacAddress(byte[] macAddress) {
        this.macAddress = macAddress;
    }

    @Override
    void appendXml(StringBuilder sb) {
        sb.append("<").append(TAG_NODE).append(">");

        sb.append("<").append(TAG_NAME);
        if (name != null && name.length() != 0) {
            sb.append(">");
            sb.append(escape(name));
            sb.append("</").append(TAG_NAME).append(">");
        } else {
            sb.append("/>");
        }
        sb.append("<").append(TAG_VENDER);
        if (vender != null && vender.length() != 0) {
            sb.append(">");
            sb.append(escape(vender));
            sb.append("</").append(TAG_VENDER).append(">");
        } else {
            sb.append("/>");
        }
        sb.append("<").append(TAG_UECSID);
        if (uecsID != null && uecsID.length() != 0) {
            sb.append(">");
            sb.append(escape(uecsID));
            sb.append("</").append(TAG_UECSID).append(">");
        } else {
            sb.append("/>");
        }
        sb.append("<").append(TAG_IP);
        if (ipAddress != null) {
            sb.append(">");
            sb.append(ipAddress.getHostAddress());
            sb.append("</").append(TAG_IP).append(">");
        } else {
            sb.append("/>");
        }
        sb.append("<").append(TAG_MAC);
        if (macAddress != null) {
            sb.append(">");
            sb.append(CommonUtils.bytesToHex(macAddress));
            sb.append("</").append(TAG_MAC).append(">");
        } else {
            sb.append("/>");
        }

        sb.append("</").append(TAG_NODE).append(">");

    }

    @Override
    void importXml(Document doc) throws Exception {
        Element elm = (Element) doc.getElementsByTagName(TAG_NAME).item(0);
        String val = elm.getFirstChild().getNodeValue();
        if (val != null && val.length() != 0) {
            setName(val);
        }
        elm = (Element) doc.getElementsByTagName(TAG_VENDER).item(0);
        val = elm.getFirstChild().getNodeValue();
        if (val != null && val.length() != 0) {
            setVender(val);
        }
        elm = (Element) doc.getElementsByTagName(TAG_UECSID).item(0);
        val = elm.getFirstChild().getNodeValue();
        if (val != null && val.length() != 0) {
            setUecsID(val);
        }
        elm = (Element) doc.getElementsByTagName(TAG_IP).item(0);
        val = elm.getFirstChild().getNodeValue();
        if (val != null && val.length() != 0) {
            ipAddress = InetAddress.getByName(val);
        }
        elm = (Element) doc.getElementsByTagName(TAG_MAC).item(0);
        val = elm.getFirstChild().getNodeValue();
        if (val != null && val.length() != 0) {
            setMacAddress(CommonUtils.hexToBytes(val));
        }
    }

    @Override
    public int getPort() {
        return UecsConstants.SCAN_PORT;
    }

    @Override
    void setXmlAttributes(String tagName, Attributes attributes) {
        // TODO Auto-generated method stub

    }

    @Override
    void setXmlValue(String tagName, String val) throws SAXException {
        if (TAG_NAME.equals(tagName)) {
            if (val != null && val.length() != 0) {
                setName(val);
            }
        }
        if (TAG_VENDER.equals(tagName)) {
            if (val != null && val.length() != 0) {
                setVender(val);
            }
        }
        if (TAG_UECSID.equals(tagName)) {
            if (val != null && val.length() != 0) {
                setUecsID(val);
            }
        }
        if (TAG_IP.equals(tagName)) {
            if (val != null && val.length() != 0) {
                try {
                    ipAddress = InetAddress.getByName(val);
                } catch (UnknownHostException e) {
                    throw new SAXException(e);
                }
            }
        }
        if (TAG_MAC.equals(tagName)) {
            if (val != null && val.length() != 0) {
                setMacAddress(CommonUtils.hexToBytes(val));
            }
        }

    }
}
