/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs.protocol;

import java.net.InetAddress;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.Attributes;

import com.wabit.uecs.CcmID;
import com.wabit.uecs.UecsConstants;

/**
 * データ送信CCMクラスです。
 *
 * @author WaBit
 *
 */
public class DataCcm extends XmlCcm {

    /** DATAタグ名 */
    public static final String TAG_DATA = "DATA";

    /** IPタグ名 */
    public static final String TAG_IP = "IP";

    private InetAddress ipAddress;

    private String value;

    @Override
    void appendXml(StringBuilder sb) {
        sb.append("<").append(TAG_DATA);
        if (getType() != null) {
            sb.append(" ").append(ATTR_TYPE).append("=\"")
                    .append(escape(getType())).append("\"");
        }
        if (getRoom() > -1) {
            sb.append(" ").append(ATTR_ROOM).append("=\"")
                    .append(getRoom()).append("\"");
        }
        if (getRegion() > -1) {
            sb.append(" ").append(ATTR_REGION).append("=\"")
                    .append(getRegion()).append("\"");
        }
        if (getOrder() > -1) {
            sb.append(" ").append(ATTR_ORDER).append("=\"")
                    .append(getOrder()).append("\"");
        }
        if (getPriority() > -1) {
            sb.append(" ").append(ATTR_PRIORITY).append("=\"").append(getPriority())
                    .append("\"");
        }
        if (getStringValue() == null) {
            sb.append("/>");
        } else {
            sb.append(">");
            sb.append(getStringValue());
            sb.append("</").append(TAG_DATA).append(">");
        }

        // version 1.0以前のCCM仕様対応
        // バージョンが空の場合はIPアドレスタグを付けない
        if (this.getUecsVersion() != null) {
            sb.append("<").append(TAG_IP);
            if (ipAddress != null) {
                sb.append(">");
                sb.append(ipAddress.getHostAddress());
                sb.append("</").append(TAG_IP).append(">");
            } else {
                sb.append("/>");
            }
        }
    }

    /**
     * 送信元ノードアドレス(IP)を取得します。
     *
     * @return IPアドレス
     */
    public InetAddress getIpAddress() {
        return ipAddress;
    }

    @Override
    void importXml(Document doc) throws Exception {
        Element elm = (Element) doc.getElementsByTagName(TAG_DATA).item(0);
        String val = elm.getAttribute(ATTR_TYPE);
        if (val.length() != 0) {
            setType(new CcmID(val));
        }
        val = elm.getAttribute(ATTR_ROOM);
        if (val.length() != 0) {
            setRoom(Integer.parseInt(val));
        } else {
            setRoom(0);
        }
        val = elm.getAttribute(ATTR_REGION);
        if (val.length() != 0) {
            setRegion(Integer.parseInt(val));
        } else {
            setRegion(0);
        }
        val = elm.getAttribute(ATTR_ORDER);
        if (val.length() != 0) {
            setOrder(Integer.parseInt(val));
        } else {
            setOrder(0);
        }
        val = elm.getAttribute(ATTR_PRIORITY);
        if (val.length() != 0) {
            setPriority(Integer.parseInt(val));
        }
        val = elm.getFirstChild().getNodeValue();
        if (val != null && val.length() != 0) {
            setValue(val);
        }

        // version 1.0以前のCCM仕様対応
        // IPアドレスタグがない場合は無視する
        if (doc.getElementsByTagName(TAG_IP).getLength() > 0) {
            elm = (Element) doc.getElementsByTagName(TAG_IP).item(0);
            val = elm.getFirstChild().getNodeValue();
            if (val != null && val.length() != 0) {
                setIpAddress(InetAddress.getByName(val));
            }
        }
    }

    /**
     * 送信元ノードアドレス(IP)を設定します。
     *
     * @param address IPアドレス
     */
    public void setIpAddress(InetAddress address) {
        this.ipAddress = address;
    }

    @Override
    public int getPort() {
        return UecsConstants.DATA_PORT;
    }

    @Override
    public String getStringValue() {
        return value;
    }

    @Override
    public Number getNumberValue() {
        if (value == null) {
            return null;
        }
        return new Double(value);
    }

    @Override
    public void setValue(Number num) {
        super.setValue(num);
        this.value = super.getStringValue();
    }

    @Override
    public void setValue(String val) {
        this.value = val;
    }

    @Override
    void setXmlAttributes(String tagName, Attributes attributes) throws Exception {

        if (TAG_DATA.equals(tagName)) {
            String val = attributes.getValue(ATTR_TYPE);
            if (val.length() != 0) {
                setType(new CcmID(val));
            }
            val = attributes.getValue(ATTR_ROOM);
            if (val.length() != 0) {
                setRoom(Integer.parseInt(val));
            } else {
                setRoom(0);
            }
            val = attributes.getValue(ATTR_REGION);
            if (val.length() != 0) {
                setRegion(Integer.parseInt(val));
            } else {
                setRegion(0);
            }
            val = attributes.getValue(ATTR_ORDER);
            if (val.length() != 0) {
                setOrder(Integer.parseInt(val));
            } else {
                setOrder(0);
            }
            val = attributes.getValue(ATTR_PRIORITY);
            if (val.length() != 0) {
                setPriority(Integer.parseInt(val));
            }

        }
    }

    @Override
    void setXmlValue(String tagName, String value) throws Exception {
        if (TAG_DATA.equals(tagName)) {
            if (value != null && value.length() != 0) {
                setValue(value);
            }
        } else if (TAG_IP.equals(tagName)) {
            // version 1.0以前のCCM仕様対応
            // IPアドレスタグがない場合は無視する
            if (value != null && value.length() != 0) {
                setIpAddress(InetAddress.getByName(value));
            }
        }
    }

}
