/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs;

import java.net.InetAddress;
import java.util.List;

import com.wabit.uecs.device.IDevice;


/**
 * UECS準拠の抽象ノードインタフェースです。
 *
 * @author WaBit
 *
 */
public interface IUecsNode<T extends NodeConfig> {

    /**
     * リスナーを登録します。
     * @param listener
     */
    public void addListener(IUecsNodeListener listener);

    /**
     * リスナー一覧を取得します。
     * @return リスナー一覧
     */
    public List<IUecsNodeListener> getListeners();

    /**
     * リスナーを削除します。
     * @param listener
     * @return リスナーが削除されるとtrue
     */
    public boolean removeListener(IUecsNodeListener listener);

    /**
     * CCMサービスを登録します。
     *
     * @param ccm CCMサービス
     */
    public void addCcmService(CcmService ccm);

    /**
     * CCMサービスを全削除します。
     */
    public void clearCcmServices();

    /**
     * 通信プロトコルを追加します。
     * @param protocol プロトコルハンドラ
     */
    public void addProtocol(IUecsProtocolHandler protocol);

    /**
     * 通信プロトコルを全削除します。
     */
    public void clearProtocols();

    /**
     * 登録されているCCMサービス数を取得します。
     *
     * @return カウント値
     */
    public int countCcm();

    /**
     * ブロードキャスト用アドレスを取得します。
     *
     * @return IPアドレス
     */
    public InetAddress getBroadcastAddress();

    /**
     * CCM通し番号を取得します。
     *
     * @param ccm 検査するCCMサービス
     * @return インデックス番号
     */
    public int getCcmIndex(CcmService ccm) ;

    /**
     * 登録されているCCMサービスを取得します。
     *
     * @param index CCM番号
     * @return ヒットしたCCMサービス
     */
    public CcmService getCcmService(int index);

    /**
     * 設定情報を取得します。
     * @return 設定値オブジェクト
     */
    public T getConfig();


    /**
     * ノードのIPアドレスを取得します。
     *
     * @return IPアドレス
     */
    public InetAddress getIpAddress();

    /**
     * ノードのMACアドレスを取得します。
     *
     * @return MACアドレス
     */
    public byte[] getMacAddress();

    /**
     * ノード名称を取得します。
     *
     * @return ノード名称
     */
    public String getName();

    /**
     * ノードのUECS IDを取得します。
     *
     * @return UECS ID
     */
    public String getUecsID();

    /**
     * UECS仕様バージョンを取得します。
     *
     * @return バージョン表記文字列
     */
    public String getUecsVersion();

    /**
     * ノードのベンダー名を取得します。
     *
     * @return ベンダー名称文字列
     */
    public String getVender();

    /**
     * 機器状態コードを取得します。
     * @return 状態コード(32ビット)
     */
    public int getStatus();

    /**
     * 機器動作モードを取得します。
     * 機器状態コード内のモードビット値領域と連動します。
     * @return 動作モード
     */
    public ActionMode getActionMode();

    /**
     * 機器動作モードを設定します。
     * 機器状態コード内のモードビット値領域と連動します。
     * @param mode 動作モード
     */
    public void setActionMode(ActionMode mode);

    /**
     * 動作状態を問い合わせます。
     *
     * @return 動作中の場合はtrue
     */
    public boolean isActive();

    /**
     * 登録されているCCMサービス一覧を取得します。
     *
     * @return サービス一覧
     */
    public List<CcmService> listCcmService();

    /**
     * CCMサービスを削除します。
     *
     * @param index インデックス番号
     * @return 削除されたCCM
     */
    public CcmService removeCcmService(int index);

    /**
     * デバイスを取得します。
     * @param deviceId デバイスID
     * @return ヒットしたデバイス
     */
    public IDevice<?> getDevice(String deviceId);

    /**
     * デバイスを登録します。
     * IDが重複するデバイスは入れ替わります。
     *
     * @param device デバイス
     */
    public void addDevice(IDevice<?> device);

    /**
     * デバイスを全削除します。
     */
    public void clearDevices();

    /**
     * 登録されているデバイス一覧を取得します。
     * @return デバイス一覧
     */
    public List<IDevice<?>> listDevices();

    /**
     * 型でフィルタ指定してデバイス一覧を取得します。
     * @param cls フィルタする型
     * @return デバイス一覧
     */
	public <U extends IDevice<?>> List<U> listDevices(Class<U> cls);

    /**
     * デバイスを削除します。
     * @param device 削除対象のデバイス
     */
    public void removeDevice(IDevice<?> device);


    /**
     * パケットを送信します。
     *
     * @param address 宛先IPアドレス
     * @param port ポート番号
     * @param data パケットデータ
     */
    public void sendPacket(InetAddress address, int port, byte[] data);

    /**
     * 機器状態コードを設定します。
     * @param status 状態コード値
     */
    public void setStatus(int status);

	/**
     * 機器状態コードをビット演算でON(1)設定します。
     * @param status 状態コード値
     */
	public void onStatus(int status);

	/**
     * 機器状態コードをビット演算でOFF(0)設定します。
     * @param status 状態コード値
     */
	public void offStatus(int status);

    /**
     * 初期設定処理を行います。
     *
     */
    public void setup();

    /**
     * サービスを開始します。
     */
    public void start();

    /**
     * サービスを停止します。
     */
    public void stop();


}
